"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const eks_generated_1 = require("./eks.generated");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: props.amiType,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_e = props.forceUpdate) !== null && _e !== void 0 ? _e : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
//# sourceMappingURL=data:application/json;base64,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