import * as autoscaling from "../../aws-autoscaling";
import * as ec2 from "../../aws-ec2";
import * as iam from "../../aws-iam";
import { Construct, IResource, Resource } from "../../core";
import { AwsAuth } from './aws-auth';
import { FargateProfile, FargateProfileOptions } from './fargate-profile';
import { HelmChart, HelmChartOptions } from './helm-chart';
import { KubernetesResource } from './k8s-resource';
import { Nodegroup, NodegroupOptions } from './managed-nodegroup';
import { ServiceAccount, ServiceAccountOptions } from './service-account';
/**
 * An EKS cluster
 */
export interface ICluster extends IResource, ec2.IConnectable {
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpc;
    /**
     * The physical name of the Cluster
     * @attribute
     */
    readonly clusterName: string;
    /**
     * The unique ARN assigned to the service by AWS
     * in the form of arn:aws:eks:
     * @attribute
     */
    readonly clusterArn: string;
    /**
     * The API Server endpoint URL
     * @attribute
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     * @attribute
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * The cluster security group that was created by Amazon EKS for the cluster.
     * @attribute
     */
    readonly clusterSecurityGroupId: string;
    /**
     * Amazon Resource Name (ARN) or alias of the customer master key (CMK).
     * @attribute
     */
    readonly clusterEncryptionConfigKeyArn: string;
}
/**
 * Attributes for EKS clusters.
 */
export interface ClusterAttributes {
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpc;
    /**
     * The physical name of the Cluster
     */
    readonly clusterName: string;
    /**
     * The unique ARN assigned to the service by AWS
     * in the form of arn:aws:eks:
     */
    readonly clusterArn: string;
    /**
     * The API Server endpoint URL
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * The cluster security group that was created by Amazon EKS for the cluster.
     */
    readonly clusterSecurityGroupId: string;
    /**
     * Amazon Resource Name (ARN) or alias of the customer master key (CMK).
     */
    readonly clusterEncryptionConfigKeyArn: string;
    /**
     * The security groups associated with this cluster.
     */
    readonly securityGroups: ec2.ISecurityGroup[];
}
/**
 * Options for configuring an EKS cluster.
 */
export interface ClusterOptions {
    /**
     * The VPC in which to create the Cluster
     *
     * @default - a VPC with default configuration will be created and can be accessed through `cluster.vpc`.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where to place EKS Control Plane ENIs
     *
     * If you want to create public load balancers, this must include public subnets.
     *
     * For example, to only select private subnets, supply the following:
     *
     * ```ts
     * vpcSubnets: [
     *   { subnetType: ec2.SubnetType.Private }
     * ]
     * ```
     *
     * @default - All public and private subnets
     */
    readonly vpcSubnets?: ec2.SubnetSelection[];
    /**
     * Role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf.
     *
     * @default - A role is automatically created for you
     */
    readonly role?: iam.IRole;
    /**
     * Name for the cluster.
     *
     * @default - Automatically generated name
     */
    readonly clusterName?: string;
    /**
     * Security Group to use for Control Plane ENIs
     *
     * @default - A security group is automatically created
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The Kubernetes version to run in the cluster
     */
    readonly version: KubernetesVersion;
    /**
     * An IAM role that will be added to the `system:masters` Kubernetes RBAC
     * group.
     *
     * @see https://kubernetes.io/docs/reference/access-authn-authz/rbac/#default-roles-and-role-bindings
     *
     * @default - By default, it will only possible to update this Kubernetes
     *            system by adding resources to this cluster via `addResource` or
     *            by defining `KubernetesResource` resources in your AWS CDK app.
     *            Use this if you wish to grant cluster administration privileges
     *            to another role.
     */
    readonly mastersRole?: iam.IRole;
    /**
     * Controls the "eks.amazonaws.com/compute-type" annotation in the CoreDNS
     * configuration on your cluster to determine which compute type to use
     * for CoreDNS.
     *
     * @default CoreDnsComputeType.EC2 (for `FargateCluster` the default is FARGATE)
     */
    readonly coreDnsComputeType?: CoreDnsComputeType;
    /**
     * Determines whether a CloudFormation output with the name of the cluster
     * will be synthesized.
     *
     * @default false
     */
    readonly outputClusterName?: boolean;
    /**
     * Determines whether a CloudFormation output with the ARN of the "masters"
     * IAM role will be synthesized (if `mastersRole` is specified).
     *
     * @default false
     */
    readonly outputMastersRoleArn?: boolean;
    /**
     * Determines whether a CloudFormation output with the `aws eks
     * update-kubeconfig` command will be synthesized. This command will include
     * the cluster name and, if applicable, the ARN of the masters IAM role.
     *
     * @default true
     */
    readonly outputConfigCommand?: boolean;
}
/**
 * Configuration props for EKS clusters.
 */
export interface ClusterProps extends ClusterOptions {
    /**
     * Allows defining `kubectrl`-related resources on this cluster.
     *
     * If this is disabled, it will not be possible to use the following
     * capabilities:
     * - `addResource`
     * - `addRoleMapping`
     * - `addUserMapping`
     * - `addMastersRole` and `props.mastersRole`
     *
     * If this is disabled, the cluster can only be managed by issuing `kubectl`
     * commands from a session that uses the IAM role/user that created the
     * account.
     *
     * _NOTE_: changing this value will destoy the cluster. This is because a
     * managable cluster must be created using an AWS CloudFormation custom
     * resource which executes with an IAM role owned by the CDK app.
     *
     * @default true The cluster can be managed by the AWS CDK application.
     */
    readonly kubectlEnabled?: boolean;
    /**
     * Number of instances to allocate as an initial capacity for this cluster.
     * Instance type can be configured through `defaultCapacityInstanceType`,
     * which defaults to `m5.large`.
     *
     * Use `cluster.addCapacity` to add additional customized capacity. Set this
     * to `0` is you wish to avoid the initial capacity allocation.
     *
     * @default 2
     */
    readonly defaultCapacity?: number;
    /**
     * The instance type to use for the default capacity. This will only be taken
     * into account if `defaultCapacity` is > 0.
     *
     * @default m5.large
     */
    readonly defaultCapacityInstance?: ec2.InstanceType;
    /**
     * The default capacity type for the cluster.
     *
     * @default NODEGROUP
     */
    readonly defaultCapacityType?: DefaultCapacityType;
}
/**
 * Kubernetes cluster version
 */
export declare class KubernetesVersion {
    readonly version: string;
    /**
     * Kubernetes version 1.14
     */
    static readonly V1_14: KubernetesVersion;
    /**
     * Kubernetes version 1.15
     */
    static readonly V1_15: KubernetesVersion;
    /**
     * Kubernetes version 1.16
     */
    static readonly V1_16: KubernetesVersion;
    /**
     * Kubernetes version 1.17
     */
    static readonly V1_17: KubernetesVersion;
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version: string): KubernetesVersion;
    /**
     *
     * @param version cluster version number
     */
    private constructor();
}
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
export declare class Cluster extends Resource implements ICluster {
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpc;
    /**
     * The Name of the created EKS Cluster
     */
    readonly clusterName: string;
    /**
     * The AWS generated ARN for the Cluster resource
     *
     * @example arn:aws:eks:us-west-2:666666666666:cluster/prod
     */
    readonly clusterArn: string;
    /**
     * The endpoint URL for the Cluster
     *
     * This is the URL inside the kubeconfig file to use with kubectl
     *
     * @example https://5E1D0CEXAMPLEA591B746AFC5AB30262.yl4.us-west-2.eks.amazonaws.com
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * The cluster security group that was created by Amazon EKS for the cluster.
     */
    readonly clusterSecurityGroupId: string;
    /**
     * Amazon Resource Name (ARN) or alias of the customer master key (CMK).
     */
    readonly clusterEncryptionConfigKeyArn: string;
    /**
     * Manages connection rules (Security Group Rules) for the cluster
     *
     * @type {ec2.Connections}
     * @memberof Cluster
     */
    readonly connections: ec2.Connections;
    /**
     * IAM role assumed by the EKS Control Plane
     */
    readonly role: iam.IRole;
    /**
     * Indicates if `kubectl` related operations can be performed on this cluster.
     */
    readonly kubectlEnabled: boolean;
    /**
     * The auto scaling group that hosts the default capacity for this cluster.
     * This will be `undefined` if the `defaultCapacityType` is not `EC2` or
     * `defaultCapacityType` is `EC2` but default capacity is set to 0.
     */
    readonly defaultCapacity?: autoscaling.AutoScalingGroup;
    /**
     * The node group that hosts the default capacity for this cluster.
     * This will be `undefined` if the `defaultCapacityType` is `EC2` or
     * `defaultCapacityType` is `NODEGROUP` but default capacity is set to 0.
     */
    readonly defaultNodegroup?: Nodegroup;
    /**
     * If the cluster has one (or more) FargateProfiles associated, this array
     * will hold a reference to each.
     */
    private readonly _fargateProfiles;
    /**
     * If this cluster is kubectl-enabled, returns the `ClusterResource` object
     * that manages it. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     */
    private readonly _clusterResource?;
    /**
     * Manages the aws-auth config map.
     */
    private _awsAuth?;
    private _openIdConnectProvider?;
    private _spotInterruptHandler?;
    private _neuronDevicePlugin?;
    private readonly version;
    /**
     * A dummy CloudFormation resource that is used as a wait barrier which
     * represents that the cluster is ready to receive "kubectl" commands.
     *
     * Specifically, all fargate profiles are automatically added as a dependency
     * of this barrier, which means that it will only be "signaled" when all
     * fargate profiles have been successfully created.
     *
     * When kubectl resources call `_attachKubectlResourceScope()`, this resource
     * is added as their dependency which implies that they can only be deployed
     * after the cluster is ready.
     */
    private readonly _kubectlReadyBarrier?;
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope: Construct, id: string, props: ClusterProps);
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id: string, options: CapacityOptions): autoscaling.AutoScalingGroup;
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id: string, options?: NodegroupOptions): Nodegroup;
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup: autoscaling.AutoScalingGroup, options: AutoScalingGroupOptions): void;
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth(): AwsAuth;
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl(): string;
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer(): string;
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider(): iam.OpenIdConnectProvider;
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id: string, ...manifest: any[]): KubernetesResource;
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id: string, options: HelmChartOptions): HelmChart;
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id: string, options: FargateProfileOptions): FargateProfile;
    /**
     * Adds a service account to this cluster.
     *
     * @param id the id of this service account
     * @param options service account options
     */
    addServiceAccount(id: string, options?: ServiceAccountOptions): ServiceAccount;
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    private addSpotInterruptHandler;
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    private addNeuronDevicePlugin;
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    private tagSubnets;
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    private defineCoreDnsComputeType;
}
/**
 * Options for adding worker nodes
 */
export interface CapacityOptions extends autoscaling.CommonAutoScalingGroupProps {
    /**
     * Instance type of the instances to start
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * Will automatically update the aws-auth ConfigMap to map the IAM instance
     * role to RBAC.
     *
     * This cannot be explicitly set to `true` if the cluster has kubectl disabled.
     *
     * @default - true if the cluster has kubectl enabled (which is the default).
     */
    readonly mapRole?: boolean;
    /**
     * Configures the EC2 user-data script for instances in this autoscaling group
     * to bootstrap the node (invoke `/etc/eks/bootstrap.sh`) and associate it
     * with the EKS cluster.
     *
     * If you wish to provide a custom user data script, set this to `false` and
     * manually invoke `autoscalingGroup.addUserData()`.
     *
     * @default true
     */
    readonly bootstrapEnabled?: boolean;
    /**
     * EKS node bootstrapping options.
     *
     * @default - none
     */
    readonly bootstrapOptions?: BootstrapOptions;
    /**
     * Machine image type
     *
     * @default MachineImageType.AMAZON_LINUX_2
     */
    readonly machineImageType?: MachineImageType;
}
/**
 * EKS node bootstrapping options.
 */
export interface BootstrapOptions {
    /**
     * Sets `--max-pods` for the kubelet based on the capacity of the EC2 instance.
     *
     * @default true
     */
    readonly useMaxPods?: boolean;
    /**
     * Restores the docker default bridge network.
     *
     * @default false
     */
    readonly enableDockerBridge?: boolean;
    /**
     * Number of retry attempts for AWS API call (DescribeCluster).
     *
     * @default 3
     */
    readonly awsApiRetryAttempts?: number;
    /**
     * The contents of the `/etc/docker/daemon.json` file. Useful if you want a
     * custom config differing from the default one in the EKS AMI.
     *
     * @default - none
     */
    readonly dockerConfigJson?: string;
    /**
     * Extra arguments to add to the kubelet. Useful for adding labels or taints.
     *
     * @example --node-labels foo=bar,goo=far
     * @default - none
     */
    readonly kubeletExtraArgs?: string;
    /**
     * Additional command line arguments to pass to the `/etc/eks/bootstrap.sh`
     * command.
     *
     * @see https://github.com/awslabs/amazon-eks-ami/blob/master/files/bootstrap.sh
     * @default - none
     */
    readonly additionalArgs?: string;
}
/**
 * Options for adding an AutoScalingGroup as capacity
 */
export interface AutoScalingGroupOptions {
    /**
     * Will automatically update the aws-auth ConfigMap to map the IAM instance
     * role to RBAC.
     *
     * This cannot be explicitly set to `true` if the cluster has kubectl disabled.
     *
     * @default - true if the cluster has kubectl enabled (which is the default).
     */
    readonly mapRole?: boolean;
    /**
     * Configures the EC2 user-data script for instances in this autoscaling group
     * to bootstrap the node (invoke `/etc/eks/bootstrap.sh`) and associate it
     * with the EKS cluster.
     *
     * If you wish to provide a custom user data script, set this to `false` and
     * manually invoke `autoscalingGroup.addUserData()`.
     *
     * @default true
     */
    readonly bootstrapEnabled?: boolean;
    /**
     * Allows options for node bootstrapping through EC2 user data.
     * @default - default options
     */
    readonly bootstrapOptions?: BootstrapOptions;
    /**
     * Allow options to specify different machine image type
     *
     * @default MachineImageType.AMAZON_LINUX_2
     */
    readonly machineImageType?: MachineImageType;
}
/**
 * Properties for EksOptimizedImage
 */
export interface EksOptimizedImageProps {
    /**
     * What instance type to retrieve the image for (standard or GPU-optimized)
     *
     * @default NodeType.STANDARD
     */
    readonly nodeType?: NodeType;
    /**
     * The Kubernetes version to use
     *
     * @default - The latest version
     */
    readonly kubernetesVersion?: string;
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
export declare class EksOptimizedImage implements ec2.IMachineImage {
    private readonly nodeType?;
    private readonly kubernetesVersion?;
    private readonly amiParameterName;
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props?: EksOptimizedImageProps);
    /**
     * Return the correct image
     */
    getImage(scope: Construct): ec2.MachineImageConfig;
}
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
export declare enum NodeType {
    /**
     * Standard instances
     */
    STANDARD = "Standard",
    /**
     * GPU instances
     */
    GPU = "GPU",
    /**
     * Inferentia instances
     */
    INFERENTIA = "INFERENTIA"
}
/**
 * The type of compute resources to use for CoreDNS.
 */
export declare enum CoreDnsComputeType {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    EC2 = "ec2",
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    FARGATE = "fargate"
}
/**
 * The default capacity type for the cluster
 */
export declare enum DefaultCapacityType {
    /**
     * managed node group
     */
    NODEGROUP = 0,
    /**
     * EC2 autoscaling group
     */
    EC2 = 1
}
/**
 * The machine image type
 */
export declare enum MachineImageType {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    AMAZON_LINUX_2 = 0,
    /**
     * Bottlerocket AMI
     */
    BOTTLEROCKET = 1
}
