"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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