"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TmpfsMountOption = exports.DevicePermission = exports.Capability = exports.LinuxParameters = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Linux-specific options that are applied to the container.
 */
class LinuxParameters extends cdk.Construct {
    /**
     * Constructs a new instance of the LinuxParameters class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * Capabilities to be added
         */
        this.capAdd = new Array();
        /**
         * Capabilities to be dropped
         */
        this.capDrop = new Array();
        /**
         * Device mounts
         */
        this.devices = new Array();
        /**
         * TmpFs mounts
         */
        this.tmpfs = new Array();
        this.sharedMemorySize = props.sharedMemorySize;
        this.initProcessEnabled = props.initProcessEnabled;
    }
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    addCapabilities(...cap) {
        this.capAdd.push(...cap);
    }
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    dropCapabilities(...cap) {
        this.capDrop.push(...cap);
    }
    /**
     * Adds one or more host devices to a container.
     */
    addDevices(...device) {
        this.devices.push(...device);
    }
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     */
    addTmpfs(...tmpfs) {
        this.tmpfs.push(...tmpfs);
    }
    /**
     * Renders the Linux parameters to a CloudFormation object.
     */
    renderLinuxParameters() {
        return {
            initProcessEnabled: this.initProcessEnabled,
            sharedMemorySize: this.sharedMemorySize,
            capabilities: {
                add: cdk.Lazy.listValue({ produce: () => this.capAdd }, { omitEmpty: true }),
                drop: cdk.Lazy.listValue({ produce: () => this.capDrop }, { omitEmpty: true }),
            },
            devices: cdk.Lazy.anyValue({ produce: () => this.devices.map(renderDevice) }, { omitEmptyArray: true }),
            tmpfs: cdk.Lazy.anyValue({ produce: () => this.tmpfs.map(renderTmpfs) }, { omitEmptyArray: true }),
        };
    }
}
exports.LinuxParameters = LinuxParameters;
function renderDevice(device) {
    return {
        containerPath: device.containerPath,
        hostPath: device.hostPath,
        permissions: device.permissions,
    };
}
function renderTmpfs(tmpfs) {
    return {
        containerPath: tmpfs.containerPath,
        size: tmpfs.size,
        mountOptions: tmpfs.mountOptions,
    };
}
/**
 * A Linux capability
 */
var Capability;
(function (Capability) {
    Capability["ALL"] = "ALL";
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    Capability["CHOWN"] = "CHOWN";
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    Capability["FOWNER"] = "FOWNER";
    Capability["FSETID"] = "FSETID";
    Capability["IPC_LOCK"] = "IPC_LOCK";
    Capability["IPC_OWNER"] = "IPC_OWNER";
    Capability["KILL"] = "KILL";
    Capability["LEASE"] = "LEASE";
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    Capability["MKNOD"] = "MKNOD";
    Capability["NET_ADMIN"] = "NET_ADMIN";
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    Capability["NET_RAW"] = "NET_RAW";
    Capability["SETFCAP"] = "SETFCAP";
    Capability["SETGID"] = "SETGID";
    Capability["SETPCAP"] = "SETPCAP";
    Capability["SETUID"] = "SETUID";
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    Capability["SYS_BOOT"] = "SYS_BOOT";
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    Capability["SYS_MODULE"] = "SYS_MODULE";
    Capability["SYS_NICE"] = "SYS_NICE";
    Capability["SYS_PACCT"] = "SYS_PACCT";
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    Capability["SYS_TIME"] = "SYS_TIME";
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    Capability["SYSLOG"] = "SYSLOG";
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
/**
 * Permissions for device access
 */
var DevicePermission;
(function (DevicePermission) {
    /**
     * Read
     */
    DevicePermission["READ"] = "read";
    /**
     * Write
     */
    DevicePermission["WRITE"] = "write";
    /**
     * Make a node
     */
    DevicePermission["MKNOD"] = "mknod";
})(DevicePermission = exports.DevicePermission || (exports.DevicePermission = {}));
/**
 * The supported options for a tmpfs mount for a container.
 */
var TmpfsMountOption;
(function (TmpfsMountOption) {
    TmpfsMountOption["DEFAULTS"] = "defaults";
    TmpfsMountOption["RO"] = "ro";
    TmpfsMountOption["RW"] = "rw";
    TmpfsMountOption["SUID"] = "suid";
    TmpfsMountOption["NOSUID"] = "nosuid";
    TmpfsMountOption["DEV"] = "dev";
    TmpfsMountOption["NODEV"] = "nodev";
    TmpfsMountOption["EXEC"] = "exec";
    TmpfsMountOption["NOEXEC"] = "noexec";
    TmpfsMountOption["SYNC"] = "sync";
    TmpfsMountOption["ASYNC"] = "async";
    TmpfsMountOption["DIRSYNC"] = "dirsync";
    TmpfsMountOption["REMOUNT"] = "remount";
    TmpfsMountOption["MAND"] = "mand";
    TmpfsMountOption["NOMAND"] = "nomand";
    TmpfsMountOption["ATIME"] = "atime";
    TmpfsMountOption["NOATIME"] = "noatime";
    TmpfsMountOption["DIRATIME"] = "diratime";
    TmpfsMountOption["NODIRATIME"] = "nodiratime";
    TmpfsMountOption["BIND"] = "bind";
    TmpfsMountOption["RBIND"] = "rbind";
    TmpfsMountOption["UNBINDABLE"] = "unbindable";
    TmpfsMountOption["RUNBINDABLE"] = "runbindable";
    TmpfsMountOption["PRIVATE"] = "private";
    TmpfsMountOption["RPRIVATE"] = "rprivate";
    TmpfsMountOption["SHARED"] = "shared";
    TmpfsMountOption["RSHARED"] = "rshared";
    TmpfsMountOption["SLAVE"] = "slave";
    TmpfsMountOption["RSLAVE"] = "rslave";
    TmpfsMountOption["RELATIME"] = "relatime";
    TmpfsMountOption["NORELATIME"] = "norelatime";
    TmpfsMountOption["STRICTATIME"] = "strictatime";
    TmpfsMountOption["NOSTRICTATIME"] = "nostrictatime";
    TmpfsMountOption["MODE"] = "mode";
    TmpfsMountOption["UID"] = "uid";
    TmpfsMountOption["GID"] = "gid";
    TmpfsMountOption["NR_INODES"] = "nr_inodes";
    TmpfsMountOption["NR_BLOCKS"] = "nr_blocks";
    TmpfsMountOption["MPOL"] = "mpol";
})(TmpfsMountOption = exports.TmpfsMountOption || (exports.TmpfsMountOption = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGludXgtcGFyYW1ldGVycy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxpbnV4LXBhcmFtZXRlcnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsa0NBQWtDLENBQUMsZ0RBQWdEO0FBbUJuRjs7R0FFRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxHQUFHLENBQUMsU0FBUztJQXlCOUM7O09BRUc7SUFDSCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLFFBQThCLEVBQUU7UUFDMUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQXBCckI7O1dBRUc7UUFDYyxXQUFNLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztRQUNsRDs7V0FFRztRQUNjLFlBQU8sR0FBRyxJQUFJLEtBQUssRUFBYyxDQUFDO1FBQ25EOztXQUVHO1FBQ2MsWUFBTyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDL0M7O1dBRUc7UUFDYyxVQUFLLEdBQUcsSUFBSSxLQUFLLEVBQVMsQ0FBQztRQU14QyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDdkQsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxlQUFlLENBQUMsR0FBRyxHQUFpQjtRQUN2QyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksZ0JBQWdCLENBQUMsR0FBRyxHQUFpQjtRQUN4QyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFDRDs7T0FFRztJQUNJLFVBQVUsQ0FBQyxHQUFHLE1BQWdCO1FBQ2pDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsTUFBTSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxRQUFRLENBQUMsR0FBRyxLQUFjO1FBQzdCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUNEOztPQUVHO0lBQ0kscUJBQXFCO1FBQ3hCLE9BQU87WUFDSCxrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCO1lBQzNDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7WUFDdkMsWUFBWSxFQUFFO2dCQUNWLEdBQUcsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7Z0JBQzVFLElBQUksRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7YUFDakY7WUFDRCxPQUFPLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBQztZQUN2RyxLQUFLLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBQztTQUNyRyxDQUFDO0lBQ04sQ0FBQztDQUNKO0FBOUVELDBDQThFQztBQXVCRCxTQUFTLFlBQVksQ0FBQyxNQUFjO0lBQ2hDLE9BQU87UUFDSCxhQUFhLEVBQUUsTUFBTSxDQUFDLGFBQWE7UUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1FBQ3pCLFdBQVcsRUFBRSxNQUFNLENBQUMsV0FBVztLQUNsQyxDQUFDO0FBQ04sQ0FBQztBQW1CRCxTQUFTLFdBQVcsQ0FBQyxLQUFZO0lBQzdCLE9BQU87UUFDSCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7UUFDbEMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1FBQ2hCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtLQUNuQyxDQUFDO0FBQ04sQ0FBQztBQUNEOztHQUVHO0FBQ0gsSUFBWSxVQXVDWDtBQXZDRCxXQUFZLFVBQVU7SUFDbEIseUJBQVcsQ0FBQTtJQUNYLDZDQUErQixDQUFBO0lBQy9CLHlDQUEyQixDQUFBO0lBQzNCLDZDQUErQixDQUFBO0lBQy9CLDZCQUFlLENBQUE7SUFDZiwyQ0FBNkIsQ0FBQTtJQUM3QixpREFBbUMsQ0FBQTtJQUNuQywrQkFBaUIsQ0FBQTtJQUNqQiwrQkFBaUIsQ0FBQTtJQUNqQixtQ0FBcUIsQ0FBQTtJQUNyQixxQ0FBdUIsQ0FBQTtJQUN2QiwyQkFBYSxDQUFBO0lBQ2IsNkJBQWUsQ0FBQTtJQUNmLGlEQUFtQyxDQUFBO0lBQ25DLHFDQUF1QixDQUFBO0lBQ3ZCLDJDQUE2QixDQUFBO0lBQzdCLDZCQUFlLENBQUE7SUFDZixxQ0FBdUIsQ0FBQTtJQUN2QixtREFBcUMsQ0FBQTtJQUNyQyw2Q0FBK0IsQ0FBQTtJQUMvQixpQ0FBbUIsQ0FBQTtJQUNuQixpQ0FBbUIsQ0FBQTtJQUNuQiwrQkFBaUIsQ0FBQTtJQUNqQixpQ0FBbUIsQ0FBQTtJQUNuQiwrQkFBaUIsQ0FBQTtJQUNqQixxQ0FBdUIsQ0FBQTtJQUN2QixtQ0FBcUIsQ0FBQTtJQUNyQix1Q0FBeUIsQ0FBQTtJQUN6Qix1Q0FBeUIsQ0FBQTtJQUN6QixtQ0FBcUIsQ0FBQTtJQUNyQixxQ0FBdUIsQ0FBQTtJQUN2Qix1Q0FBeUIsQ0FBQTtJQUN6QixxQ0FBdUIsQ0FBQTtJQUN2QiwyQ0FBNkIsQ0FBQTtJQUM3QixtQ0FBcUIsQ0FBQTtJQUNyQiwrQ0FBaUMsQ0FBQTtJQUNqQywrQkFBaUIsQ0FBQTtJQUNqQix1Q0FBeUIsQ0FBQTtBQUM3QixDQUFDLEVBdkNXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBdUNyQjtBQUNEOztHQUVHO0FBQ0gsSUFBWSxnQkFhWDtBQWJELFdBQVksZ0JBQWdCO0lBQ3hCOztPQUVHO0lBQ0gsaUNBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsbUNBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsbUNBQWUsQ0FBQTtBQUNuQixDQUFDLEVBYlcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFhM0I7QUFDRDs7R0FFRztBQUNILElBQVksZ0JBd0NYO0FBeENELFdBQVksZ0JBQWdCO0lBQ3hCLHlDQUFxQixDQUFBO0lBQ3JCLDZCQUFTLENBQUE7SUFDVCw2QkFBUyxDQUFBO0lBQ1QsaUNBQWEsQ0FBQTtJQUNiLHFDQUFpQixDQUFBO0lBQ2pCLCtCQUFXLENBQUE7SUFDWCxtQ0FBZSxDQUFBO0lBQ2YsaUNBQWEsQ0FBQTtJQUNiLHFDQUFpQixDQUFBO0lBQ2pCLGlDQUFhLENBQUE7SUFDYixtQ0FBZSxDQUFBO0lBQ2YsdUNBQW1CLENBQUE7SUFDbkIsdUNBQW1CLENBQUE7SUFDbkIsaUNBQWEsQ0FBQTtJQUNiLHFDQUFpQixDQUFBO0lBQ2pCLG1DQUFlLENBQUE7SUFDZix1Q0FBbUIsQ0FBQTtJQUNuQix5Q0FBcUIsQ0FBQTtJQUNyQiw2Q0FBeUIsQ0FBQTtJQUN6QixpQ0FBYSxDQUFBO0lBQ2IsbUNBQWUsQ0FBQTtJQUNmLDZDQUF5QixDQUFBO0lBQ3pCLCtDQUEyQixDQUFBO0lBQzNCLHVDQUFtQixDQUFBO0lBQ25CLHlDQUFxQixDQUFBO0lBQ3JCLHFDQUFpQixDQUFBO0lBQ2pCLHVDQUFtQixDQUFBO0lBQ25CLG1DQUFlLENBQUE7SUFDZixxQ0FBaUIsQ0FBQTtJQUNqQix5Q0FBcUIsQ0FBQTtJQUNyQiw2Q0FBeUIsQ0FBQTtJQUN6QiwrQ0FBMkIsQ0FBQTtJQUMzQixtREFBK0IsQ0FBQTtJQUMvQixpQ0FBYSxDQUFBO0lBQ2IsK0JBQVcsQ0FBQTtJQUNYLCtCQUFXLENBQUE7SUFDWCwyQ0FBdUIsQ0FBQTtJQUN2QiwyQ0FBdUIsQ0FBQTtJQUN2QixpQ0FBYSxDQUFBO0FBQ2pCLENBQUMsRUF4Q1csZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUF3QzNCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ2ZuVGFza0RlZmluaXRpb24gfSBmcm9tICcuL2Vjcy5nZW5lcmF0ZWQnO1xuLyoqXG4gKiBUaGUgcHJvcGVydGllcyBmb3IgZGVmaW5pbmcgTGludXgtc3BlY2lmaWMgb3B0aW9ucyB0aGF0IGFyZSBhcHBsaWVkIHRvIHRoZSBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGludXhQYXJhbWV0ZXJzUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB3aGV0aGVyIHRvIHJ1biBhbiBpbml0IHByb2Nlc3MgaW5zaWRlIHRoZSBjb250YWluZXIgdGhhdCBmb3J3YXJkcyBzaWduYWxzIGFuZCByZWFwcyBwcm9jZXNzZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluaXRQcm9jZXNzRW5hYmxlZD86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlIGZvciB0aGUgc2l6ZSAoaW4gTWlCKSBvZiB0aGUgL2Rldi9zaG0gdm9sdW1lLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gc2hhcmVkIG1lbW9yeS5cbiAgICAgKi9cbiAgICByZWFkb25seSBzaGFyZWRNZW1vcnlTaXplPzogbnVtYmVyO1xufVxuLyoqXG4gKiBMaW51eC1zcGVjaWZpYyBvcHRpb25zIHRoYXQgYXJlIGFwcGxpZWQgdG8gdGhlIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGNsYXNzIExpbnV4UGFyYW1ldGVycyBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIGluaXQgcHJvY2VzcyBpcyBlbmFibGVkXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBpbml0UHJvY2Vzc0VuYWJsZWQ/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBzaGFyZWQgbWVtb3J5IHNpemUuIE5vdCB2YWxpZCBmb3IgRmFyZ2F0ZSBsYXVuY2ggdHlwZVxuICAgICAqL1xuICAgIHByaXZhdGUgcmVhZG9ubHkgc2hhcmVkTWVtb3J5U2l6ZT86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBDYXBhYmlsaXRpZXMgdG8gYmUgYWRkZWRcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGNhcEFkZCA9IG5ldyBBcnJheTxDYXBhYmlsaXR5PigpO1xuICAgIC8qKlxuICAgICAqIENhcGFiaWxpdGllcyB0byBiZSBkcm9wcGVkXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBjYXBEcm9wID0gbmV3IEFycmF5PENhcGFiaWxpdHk+KCk7XG4gICAgLyoqXG4gICAgICogRGV2aWNlIG1vdW50c1xuICAgICAqL1xuICAgIHByaXZhdGUgcmVhZG9ubHkgZGV2aWNlcyA9IG5ldyBBcnJheTxEZXZpY2U+KCk7XG4gICAgLyoqXG4gICAgICogVG1wRnMgbW91bnRzXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSB0bXBmcyA9IG5ldyBBcnJheTxUbXBmcz4oKTtcbiAgICAvKipcbiAgICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBMaW51eFBhcmFtZXRlcnMgY2xhc3MuXG4gICAgICovXG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBMaW51eFBhcmFtZXRlcnNQcm9wcyA9IHt9KSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuc2hhcmVkTWVtb3J5U2l6ZSA9IHByb3BzLnNoYXJlZE1lbW9yeVNpemU7XG4gICAgICAgIHRoaXMuaW5pdFByb2Nlc3NFbmFibGVkID0gcHJvcHMuaW5pdFByb2Nlc3NFbmFibGVkO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIG9uZSBvciBtb3JlIExpbnV4IGNhcGFiaWxpdGllcyB0byB0aGUgRG9ja2VyIGNvbmZpZ3VyYXRpb24gb2YgYSBjb250YWluZXIuXG4gICAgICpcbiAgICAgKiBPbmx5IHdvcmtzIHdpdGggRUMyIGxhdW5jaCB0eXBlLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRDYXBhYmlsaXRpZXMoLi4uY2FwOiBDYXBhYmlsaXR5W10pIHtcbiAgICAgICAgdGhpcy5jYXBBZGQucHVzaCguLi5jYXApO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZW1vdmVzIG9uZSBvciBtb3JlIExpbnV4IGNhcGFiaWxpdGllcyB0byB0aGUgRG9ja2VyIGNvbmZpZ3VyYXRpb24gb2YgYSBjb250YWluZXIuXG4gICAgICpcbiAgICAgKiBPbmx5IHdvcmtzIHdpdGggRUMyIGxhdW5jaCB0eXBlLlxuICAgICAqL1xuICAgIHB1YmxpYyBkcm9wQ2FwYWJpbGl0aWVzKC4uLmNhcDogQ2FwYWJpbGl0eVtdKSB7XG4gICAgICAgIHRoaXMuY2FwRHJvcC5wdXNoKC4uLmNhcCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgb25lIG9yIG1vcmUgaG9zdCBkZXZpY2VzIHRvIGEgY29udGFpbmVyLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGREZXZpY2VzKC4uLmRldmljZTogRGV2aWNlW10pIHtcbiAgICAgICAgdGhpcy5kZXZpY2VzLnB1c2goLi4uZGV2aWNlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIHRoZSBjb250YWluZXIgcGF0aCwgbW91bnQgb3B0aW9ucywgYW5kIHNpemUgKGluIE1pQikgb2YgdGhlIHRtcGZzIG1vdW50IGZvciBhIGNvbnRhaW5lci5cbiAgICAgKlxuICAgICAqIE9ubHkgd29ya3Mgd2l0aCBFQzIgbGF1bmNoIHR5cGUuXG4gICAgICovXG4gICAgcHVibGljIGFkZFRtcGZzKC4uLnRtcGZzOiBUbXBmc1tdKSB7XG4gICAgICAgIHRoaXMudG1wZnMucHVzaCguLi50bXBmcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlbmRlcnMgdGhlIExpbnV4IHBhcmFtZXRlcnMgdG8gYSBDbG91ZEZvcm1hdGlvbiBvYmplY3QuXG4gICAgICovXG4gICAgcHVibGljIHJlbmRlckxpbnV4UGFyYW1ldGVycygpOiBDZm5UYXNrRGVmaW5pdGlvbi5MaW51eFBhcmFtZXRlcnNQcm9wZXJ0eSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBpbml0UHJvY2Vzc0VuYWJsZWQ6IHRoaXMuaW5pdFByb2Nlc3NFbmFibGVkLFxuICAgICAgICAgICAgc2hhcmVkTWVtb3J5U2l6ZTogdGhpcy5zaGFyZWRNZW1vcnlTaXplLFxuICAgICAgICAgICAgY2FwYWJpbGl0aWVzOiB7XG4gICAgICAgICAgICAgICAgYWRkOiBjZGsuTGF6eS5saXN0VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmNhcEFkZCB9LCB7IG9taXRFbXB0eTogdHJ1ZSB9KSxcbiAgICAgICAgICAgICAgICBkcm9wOiBjZGsuTGF6eS5saXN0VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmNhcERyb3AgfSwgeyBvbWl0RW1wdHk6IHRydWUgfSksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgZGV2aWNlczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmRldmljZXMubWFwKHJlbmRlckRldmljZSkgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9KSxcbiAgICAgICAgICAgIHRtcGZzOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMudG1wZnMubWFwKHJlbmRlclRtcGZzKSB9LCB7IG9taXRFbXB0eUFycmF5OiB0cnVlIH0pLFxuICAgICAgICB9O1xuICAgIH1cbn1cbi8qKlxuICogQSBjb250YWluZXIgaW5zdGFuY2UgaG9zdCBkZXZpY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGV2aWNlIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcGF0aCBpbnNpZGUgdGhlIGNvbnRhaW5lciBhdCB3aGljaCB0byBleHBvc2UgdGhlIGhvc3QgZGV2aWNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgU2FtZSBwYXRoIGFzIHRoZSBob3N0XG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGFpbmVyUGF0aD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcGF0aCBmb3IgdGhlIGRldmljZSBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaG9zdFBhdGg6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgZXhwbGljaXQgcGVybWlzc2lvbnMgdG8gcHJvdmlkZSB0byB0aGUgY29udGFpbmVyIGZvciB0aGUgZGV2aWNlLlxuICAgICAqIEJ5IGRlZmF1bHQsIHRoZSBjb250YWluZXIgaGFzIHBlcm1pc3Npb25zIGZvciByZWFkLCB3cml0ZSwgYW5kIG1rbm9kIGZvciB0aGUgZGV2aWNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgUmVhZG9ubHlcbiAgICAgKi9cbiAgICByZWFkb25seSBwZXJtaXNzaW9ucz86IERldmljZVBlcm1pc3Npb25bXTtcbn1cbmZ1bmN0aW9uIHJlbmRlckRldmljZShkZXZpY2U6IERldmljZSk6IENmblRhc2tEZWZpbml0aW9uLkRldmljZVByb3BlcnR5IHtcbiAgICByZXR1cm4ge1xuICAgICAgICBjb250YWluZXJQYXRoOiBkZXZpY2UuY29udGFpbmVyUGF0aCxcbiAgICAgICAgaG9zdFBhdGg6IGRldmljZS5ob3N0UGF0aCxcbiAgICAgICAgcGVybWlzc2lvbnM6IGRldmljZS5wZXJtaXNzaW9ucyxcbiAgICB9O1xufVxuLyoqXG4gKiBUaGUgZGV0YWlscyBvZiBhIHRtcGZzIG1vdW50IGZvciBhIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUbXBmcyB7XG4gICAgLyoqXG4gICAgICogVGhlIGFic29sdXRlIGZpbGUgcGF0aCB3aGVyZSB0aGUgdG1wZnMgdm9sdW1lIGlzIHRvIGJlIG1vdW50ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGFpbmVyUGF0aDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBzaXplIChpbiBNaUIpIG9mIHRoZSB0bXBmcyB2b2x1bWUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2l6ZTogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBsaXN0IG9mIHRtcGZzIHZvbHVtZSBtb3VudCBvcHRpb25zLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAgICogW1RtcGZzTW91bnRPcHRpb25zXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RtcGZzLmh0bWwpLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1vdW50T3B0aW9ucz86IFRtcGZzTW91bnRPcHRpb25bXTtcbn1cbmZ1bmN0aW9uIHJlbmRlclRtcGZzKHRtcGZzOiBUbXBmcyk6IENmblRhc2tEZWZpbml0aW9uLlRtcGZzUHJvcGVydHkge1xuICAgIHJldHVybiB7XG4gICAgICAgIGNvbnRhaW5lclBhdGg6IHRtcGZzLmNvbnRhaW5lclBhdGgsXG4gICAgICAgIHNpemU6IHRtcGZzLnNpemUsXG4gICAgICAgIG1vdW50T3B0aW9uczogdG1wZnMubW91bnRPcHRpb25zLFxuICAgIH07XG59XG4vKipcbiAqIEEgTGludXggY2FwYWJpbGl0eVxuICovXG5leHBvcnQgZW51bSBDYXBhYmlsaXR5IHtcbiAgICBBTEwgPSAnQUxMJyxcbiAgICBBVURJVF9DT05UUk9MID0gJ0FVRElUX0NPTlRST0wnLFxuICAgIEFVRElUX1dSSVRFID0gJ0FVRElUX1dSSVRFJyxcbiAgICBCTE9DS19TVVNQRU5EID0gJ0JMT0NLX1NVU1BFTkQnLFxuICAgIENIT1dOID0gJ0NIT1dOJyxcbiAgICBEQUNfT1ZFUlJJREUgPSAnREFDX09WRVJSSURFJyxcbiAgICBEQUNfUkVBRF9TRUFSQ0ggPSAnREFDX1JFQURfU0VBUkNIJyxcbiAgICBGT1dORVIgPSAnRk9XTkVSJyxcbiAgICBGU0VUSUQgPSAnRlNFVElEJyxcbiAgICBJUENfTE9DSyA9ICdJUENfTE9DSycsXG4gICAgSVBDX09XTkVSID0gJ0lQQ19PV05FUicsXG4gICAgS0lMTCA9ICdLSUxMJyxcbiAgICBMRUFTRSA9ICdMRUFTRScsXG4gICAgTElOVVhfSU1NVVRBQkxFID0gJ0xJTlVYX0lNTVVUQUJMRScsXG4gICAgTUFDX0FETUlOID0gJ01BQ19BRE1JTicsXG4gICAgTUFDX09WRVJSSURFID0gJ01BQ19PVkVSUklERScsXG4gICAgTUtOT0QgPSAnTUtOT0QnLFxuICAgIE5FVF9BRE1JTiA9ICdORVRfQURNSU4nLFxuICAgIE5FVF9CSU5EX1NFUlZJQ0UgPSAnTkVUX0JJTkRfU0VSVklDRScsXG4gICAgTkVUX0JST0FEQ0FTVCA9ICdORVRfQlJPQURDQVNUJyxcbiAgICBORVRfUkFXID0gJ05FVF9SQVcnLFxuICAgIFNFVEZDQVAgPSAnU0VURkNBUCcsXG4gICAgU0VUR0lEID0gJ1NFVEdJRCcsXG4gICAgU0VUUENBUCA9ICdTRVRQQ0FQJyxcbiAgICBTRVRVSUQgPSAnU0VUVUlEJyxcbiAgICBTWVNfQURNSU4gPSAnU1lTX0FETUlOJyxcbiAgICBTWVNfQk9PVCA9ICdTWVNfQk9PVCcsXG4gICAgU1lTX0NIUk9PVCA9ICdTWVNfQ0hST09UJyxcbiAgICBTWVNfTU9EVUxFID0gJ1NZU19NT0RVTEUnLFxuICAgIFNZU19OSUNFID0gJ1NZU19OSUNFJyxcbiAgICBTWVNfUEFDQ1QgPSAnU1lTX1BBQ0NUJyxcbiAgICBTWVNfUFRSQUNFID0gJ1NZU19QVFJBQ0UnLFxuICAgIFNZU19SQVdJTyA9ICdTWVNfUkFXSU8nLFxuICAgIFNZU19SRVNPVVJDRSA9ICdTWVNfUkVTT1VSQ0UnLFxuICAgIFNZU19USU1FID0gJ1NZU19USU1FJyxcbiAgICBTWVNfVFRZX0NPTkZJRyA9ICdTWVNfVFRZX0NPTkZJRycsXG4gICAgU1lTTE9HID0gJ1NZU0xPRycsXG4gICAgV0FLRV9BTEFSTSA9ICdXQUtFX0FMQVJNJ1xufVxuLyoqXG4gKiBQZXJtaXNzaW9ucyBmb3IgZGV2aWNlIGFjY2Vzc1xuICovXG5leHBvcnQgZW51bSBEZXZpY2VQZXJtaXNzaW9uIHtcbiAgICAvKipcbiAgICAgKiBSZWFkXG4gICAgICovXG4gICAgUkVBRCA9ICdyZWFkJyxcbiAgICAvKipcbiAgICAgKiBXcml0ZVxuICAgICAqL1xuICAgIFdSSVRFID0gJ3dyaXRlJyxcbiAgICAvKipcbiAgICAgKiBNYWtlIGEgbm9kZVxuICAgICAqL1xuICAgIE1LTk9EID0gJ21rbm9kJ1xufVxuLyoqXG4gKiBUaGUgc3VwcG9ydGVkIG9wdGlvbnMgZm9yIGEgdG1wZnMgbW91bnQgZm9yIGEgY29udGFpbmVyLlxuICovXG5leHBvcnQgZW51bSBUbXBmc01vdW50T3B0aW9uIHtcbiAgICBERUZBVUxUUyA9ICdkZWZhdWx0cycsXG4gICAgUk8gPSAncm8nLFxuICAgIFJXID0gJ3J3JyxcbiAgICBTVUlEID0gJ3N1aWQnLFxuICAgIE5PU1VJRCA9ICdub3N1aWQnLFxuICAgIERFViA9ICdkZXYnLFxuICAgIE5PREVWID0gJ25vZGV2JyxcbiAgICBFWEVDID0gJ2V4ZWMnLFxuICAgIE5PRVhFQyA9ICdub2V4ZWMnLFxuICAgIFNZTkMgPSAnc3luYycsXG4gICAgQVNZTkMgPSAnYXN5bmMnLFxuICAgIERJUlNZTkMgPSAnZGlyc3luYycsXG4gICAgUkVNT1VOVCA9ICdyZW1vdW50JyxcbiAgICBNQU5EID0gJ21hbmQnLFxuICAgIE5PTUFORCA9ICdub21hbmQnLFxuICAgIEFUSU1FID0gJ2F0aW1lJyxcbiAgICBOT0FUSU1FID0gJ25vYXRpbWUnLFxuICAgIERJUkFUSU1FID0gJ2RpcmF0aW1lJyxcbiAgICBOT0RJUkFUSU1FID0gJ25vZGlyYXRpbWUnLFxuICAgIEJJTkQgPSAnYmluZCcsXG4gICAgUkJJTkQgPSAncmJpbmQnLFxuICAgIFVOQklOREFCTEUgPSAndW5iaW5kYWJsZScsXG4gICAgUlVOQklOREFCTEUgPSAncnVuYmluZGFibGUnLFxuICAgIFBSSVZBVEUgPSAncHJpdmF0ZScsXG4gICAgUlBSSVZBVEUgPSAncnByaXZhdGUnLFxuICAgIFNIQVJFRCA9ICdzaGFyZWQnLFxuICAgIFJTSEFSRUQgPSAncnNoYXJlZCcsXG4gICAgU0xBVkUgPSAnc2xhdmUnLFxuICAgIFJTTEFWRSA9ICdyc2xhdmUnLFxuICAgIFJFTEFUSU1FID0gJ3JlbGF0aW1lJyxcbiAgICBOT1JFTEFUSU1FID0gJ25vcmVsYXRpbWUnLFxuICAgIFNUUklDVEFUSU1FID0gJ3N0cmljdGF0aW1lJyxcbiAgICBOT1NUUklDVEFUSU1FID0gJ25vc3RyaWN0YXRpbWUnLFxuICAgIE1PREUgPSAnbW9kZScsXG4gICAgVUlEID0gJ3VpZCcsXG4gICAgR0lEID0gJ2dpZCcsXG4gICAgTlJfSU5PREVTID0gJ25yX2lub2RlcycsXG4gICAgTlJfQkxPQ0tTID0gJ25yX2Jsb2NrcycsXG4gICAgTVBPTCA9ICdtcG9sJ1xufVxuIl19