"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuiltInAttributes = exports.Ec2Service = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
const task_definition_1 = require("../base/task-definition");
/**
 * This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class Ec2Service extends base_service_1.BaseService {
    /**
     * Constructs a new instance of the Ec2Service class.
     */
    constructor(scope, id, props) {
        var _a;
        if (props.daemon && props.desiredCount !== undefined) {
            throw new Error('Daemon mode launches one task on every instance. Don\'t supply desiredCount.');
        }
        if (props.daemon && props.maxHealthyPercent !== undefined && props.maxHealthyPercent !== 100) {
            throw new Error('Maximum percent must be 100 for daemon mode.');
        }
        if (props.minHealthyPercent !== undefined && props.maxHealthyPercent !== undefined && props.minHealthyPercent >= props.maxHealthyPercent) {
            throw new Error('Minimum healthy percent must be less than maximum healthy percent.');
        }
        if (!props.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with EC2');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            // If daemon, desiredCount must be undefined and that's what we want. Otherwise, default to 1.
            desiredCount: props.daemon || props.desiredCount !== undefined ? props.desiredCount : 1,
            maxHealthyPercent: props.daemon && props.maxHealthyPercent === undefined ? 100 : props.maxHealthyPercent,
            minHealthyPercent: props.daemon && props.minHealthyPercent === undefined ? 0 : props.minHealthyPercent,
            launchType: base_service_1.LaunchType.EC2,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            placementConstraints: core_1.Lazy.anyValue({ produce: () => this.constraints }, { omitEmptyArray: true }),
            placementStrategies: core_1.Lazy.anyValue({ produce: () => this.strategies }, { omitEmptyArray: true }),
            schedulingStrategy: props.daemon ? 'DAEMON' : 'REPLICA',
        }, props.taskDefinition);
        this.constraints = [];
        this.strategies = [];
        this.daemon = props.daemon || false;
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        if (props.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroups from ASG.
            // We have to be smart here -- by default future Security Group rules would be created
            // in the Cluster stack. However, if the Cluster is in a different stack than us,
            // that will lead to a cyclic reference (we point to that stack for the cluster name,
            // but that stack will point to the ALB probably created right next to us).
            //
            // In that case, reference the same security groups but make sure new rules are
            // created in the current scope (i.e., this stack)
            validateNoNetworkingProps(props);
            this.connections.addSecurityGroup(...securityGroupsInThisStack(this, props.cluster.connections.securityGroups));
        }
        this.addPlacementConstraints(...props.placementConstraints || []);
        this.addPlacementStrategies(...props.placementStrategies || []);
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
    /**
     * Imports from the specified service ARN.
     */
    static fromEc2ServiceArn(scope, id, ec2ServiceArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = ec2ServiceArn;
                this.serviceName = core_1.Stack.of(scope).parseArn(ec2ServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports from the specified service attrributes.
     */
    static fromEc2ServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     */
    addPlacementStrategies(...strategies) {
        if (strategies.length > 0 && this.daemon) {
            throw new Error("Can't configure placement strategies when daemon=true");
        }
        for (const strategy of strategies) {
            this.strategies.push(...strategy.toJson());
        }
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     */
    addPlacementConstraints(...constraints) {
        for (const constraint of constraints) {
            this.constraints.push(...constraint.toJson());
        }
    }
    /**
     * Validates this Ec2Service.
     */
    validate() {
        const ret = super.validate();
        if (!this.cluster.hasEc2Capacity) {
            ret.push('Cluster for this service needs Ec2 capacity. Call addXxxCapacity() on the cluster.');
        }
        return ret;
    }
}
exports.Ec2Service = Ec2Service;
/**
 * Validate combinations of networking arguments.
 */
function validateNoNetworkingProps(props) {
    if (props.vpcSubnets !== undefined
        || props.securityGroup !== undefined
        || props.securityGroups !== undefined
        || props.assignPublicIp) {
        throw new Error('vpcSubnets, securityGroup(s) and assignPublicIp can only be used in AwsVpc networking mode');
    }
}
/**
 * Force security group rules to be created in this stack.
 *
 * For every security group, if the scope and the group are in different stacks, return
 * a fake "imported" security group instead. This will behave as the original security group,
 * but new Ingress and Egress rule resources will be added in the current stack instead of the
 * other one.
 */
function securityGroupsInThisStack(scope, groups) {
    const thisStack = core_1.Stack.of(scope);
    let i = 1;
    return groups.map(group => {
        if (thisStack === core_1.Stack.of(group)) {
            return group;
        } // Simple case, just return the original one
        return ec2.SecurityGroup.fromSecurityGroupId(scope, `SecurityGroup${i++}`, group.securityGroupId, {
            allowAllOutbound: group.allowAllOutbound,
            mutable: true,
        });
    });
}
/**
 * The built-in container instance attributes
 */
class BuiltInAttributes {
}
exports.BuiltInAttributes = BuiltInAttributes;
/**
 * The id of the instance.
 */
BuiltInAttributes.INSTANCE_ID = 'instanceId';
/**
 * The AvailabilityZone where the instance is running in.
 */
BuiltInAttributes.AVAILABILITY_ZONE = 'attribute:ecs.availability-zone';
/**
 * The AMI id the instance is using.
 */
BuiltInAttributes.AMI_ID = 'attribute:ecs.ami-id';
/**
 * The EC2 instance type.
 */
BuiltInAttributes.INSTANCE_TYPE = 'attribute:ecs.instance-type';
/**
 * The operating system of the instance.
 *
 * Either 'linux' or 'windows'.
 */
BuiltInAttributes.OS_TYPE = 'attribute:ecs.os-type';
//# sourceMappingURL=data:application/json;base64,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