"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.ContainerDependencyCondition = exports.UlimitName = exports.ContainerDefinition = exports.Secret = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("./base/task-definition");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManager(secret, field) {
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends cdk.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The configured container links
         */
        this.links = new Array();
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential !== undefined ? props.essential : true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = this.node.id;
        this.imageConfig = props.image.bind(this, this);
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        props.taskDefinition._linkContainer(this);
        if (props.secrets) {
            this.secrets = [];
            for (const [name, secret] of Object.entries(props.secrets)) {
                if (this.taskDefinition.isFargateCompatible && secret.hasField) {
                    throw new Error(`Cannot specify secret JSON field for a task using the FARGATE launch type: '${name}' in container '${this.node.id}'`);
                }
                secret.grantRead(this.taskDefinition.obtainExecutionRole());
                this.secrets.push({
                    name,
                    valueFrom: secret.arn,
                });
            }
        }
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = {
                        ...pm,
                        hostPort: 0,
                    };
                }
            }
            return pm;
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     */
    findPortMapping(containerPort, protocol) {
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change)
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.anyValue({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.anyValue({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.anyValue({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.anyValue({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.anyValue({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.props.environment && renderKV(this.props.environment, 'name', 'value'),
            secrets: this.secrets,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.listValue({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (this.props.gpuCount !== undefined) ? renderResourceRequirements(this.props.gpuCount) : undefined,
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = new Array();
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval != null ? hc.interval.toSeconds() : 30,
        retries: hc.retries !== undefined ? hc.retries : 3,
        startPeriod: hc.startPeriod && hc.startPeriod.toSeconds(),
        timeout: hc.timeout !== undefined ? hc.timeout.toSeconds() : 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount) {
    if (gpuCount === 0) {
        return undefined;
    }
    return [{
            type: 'GPU',
            value: gpuCount.toString(),
        }];
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
//# sourceMappingURL=data:application/json;base64,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