"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsServiceBase = void 0;
const aws_certificatemanager_1 = require("../../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The base class for ApplicationMultipleTargetGroupsEc2Service and ApplicationMultipleTargetGroupsFargateService classes.
 */
class ApplicationMultipleTargetGroupsServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                const protocolType = new Set();
                for (const listenerProps of lbProps.listeners) {
                    const protocol = this.createListenerProtocol(listenerProps.protocol, listenerProps.certificate);
                    if (listenerProps.certificate !== undefined && protocol !== undefined && protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                        throw new Error('The HTTPS protocol must be used when a certificate is given');
                    }
                    protocolType.add(protocol);
                    const listener = this.configListener(protocol, {
                        certificate: listenerProps.certificate,
                        domainName: lbProps.domainName,
                        domainZone: lbProps.domainZone,
                        listenerName: listenerProps.name,
                        loadBalancer: lb,
                        port: listenerProps.port,
                    });
                    this.listeners.push(listener);
                }
                const domainName = this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
                for (const protocol of protocolType) {
                    new core_1.CfnOutput(this, `ServiceURL${lb.node.id}${protocol.toLowerCase()}`, { value: protocol.toLowerCase() + '://' + domainName });
                }
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            const protocol = this.createListenerProtocol();
            this.listener = this.configListener(protocol, {
                listenerName: 'PublicListener',
                loadBalancer: this.loadBalancer,
            });
            const domainName = this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
            new core_1.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                        protocol: targetProps.protocol,
                    }),
                ],
                hostHeader: targetProps.hostHeader,
                pathPattern: targetProps.pathPattern,
                priority: targetProps.priority,
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, target.protocol || aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                    protocol: target.protocol,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    configListener(protocol, props) {
        const listener = this.createListener(props.listenerName, props.loadBalancer, protocol, props.port);
        let certificate;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            certificate = this.createListenerCertificate(props.listenerName, props.certificate, props.domainName, props.domainZone);
        }
        else {
            certificate = undefined;
        }
        if (certificate !== undefined) {
            listener.addCertificateArns(`Arns${props.listenerName}`, [certificate.certificateArn]);
        }
        return listener;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, name, lbProps);
    }
    createListenerProtocol(listenerProtocol, certificate) {
        return listenerProtocol !== undefined ? listenerProtocol : (certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
    }
    createListenerCertificate(listenerName, certificate, domainName, domainZone) {
        if (typeof domainName === 'undefined' || typeof domainZone === 'undefined') {
            throw new Error('A domain name and zone is required when using the HTTPS protocol');
        }
        if (certificate !== undefined) {
            return certificate;
        }
        else {
            return new aws_certificatemanager_1.Certificate(this, `Certificate${listenerName}`, {
                domainName,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(domainZone),
            });
        }
    }
    createListener(name, lb, protocol, port) {
        return lb.addListener(name, {
            protocol,
            open: true,
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        return domainName;
    }
}
exports.ApplicationMultipleTargetGroupsServiceBase = ApplicationMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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