"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = exports.RepositoryBase = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cr = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        const parts = this.stack.parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${tagSuffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed
     *
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onImageScanCompleted(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': options.imageTags ? options.imageTags : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.anyValue({ produce: () => this.renderLifecyclePolicy() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
        // image scanOnPush
        if (props.imageScanOnPush) {
            new cr.AwsCustomResource(this, 'ImageScanOnPush', {
                resourceType: 'Custom::ECRImageScanOnPush',
                onUpdate: {
                    service: 'ECR',
                    action: 'putImageScanningConfiguration',
                    parameters: {
                        repositoryName: this.repositoryName,
                        imageScanningConfiguration: {
                            scanOnPush: props.imageScanOnPush,
                        },
                    },
                    physicalResourceId: cr.PhysicalResourceId.of(this.repositoryArn),
                },
                onDelete: {
                    service: 'ECR',
                    action: 'putImageScanningConfiguration',
                    parameters: {
                        repositoryName: this.repositoryName,
                        imageScanningConfiguration: {
                            scanOnPush: false,
                        },
                    },
                    physicalResourceId: cr.PhysicalResourceId.of(this.repositoryArn),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryArn] }),
            });
        }
    }
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope) {
        return core_1.Stack.of(scope).formatArn({
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: rule.rulePriority !== undefined ? rule.rulePriority : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: rule.maxImageAge !== undefined ? rule.maxImageAge.toDays() : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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