"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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