"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _a;
        const accountRecovery = (_a = props.accountRecovery) !== null && _a !== void 0 ? _a : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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