"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 ? callbackUrls : undefined,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        var _a, _b, _c, _d, _e;
        const authFlows = [];
        if ((_a = props.authFlows) === null || _a === void 0 ? void 0 : _a.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if ((_b = props.authFlows) === null || _b === void 0 ? void 0 : _b.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if ((_c = props.authFlows) === null || _c === void 0 ? void 0 : _c.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if ((_d = props.authFlows) === null || _d === void 0 ? void 0 : _d.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        if ((_e = props.authFlows) === null || _e === void 0 ? void 0 : _e.refreshToken) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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