"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = stage.pipeline.node.uniqueId + 'SourceEventRule' + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(id)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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