"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeployAction = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: this.props.imageFile && this.props.imageFile.fileName,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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