"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProfilingGroup = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.profilingGroupName) !== null && _a !== void 0 ? _a : core_1.Lazy.stringValue({ produce: () => this.generateUniqueId() }),
        });
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).parseArn(profilingGroupArn).resource;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = this.node.uniqueId;
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.ProfilingGroup = ProfilingGroup;
//# sourceMappingURL=data:application/json;base64,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