"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeBuildSpecs = exports.BuildSpec = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * BuildSpec for CodeBuild projects
 */
class BuildSpec {
    constructor() {
    }
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2),
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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