"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: {
                comment: (props && props.comment) || 'Allows CloudFront to reach the bucket',
            },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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