"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(props) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = props.domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket) {
        if (bucket.isWebsite) {
            return new HttpOrigin({
                domainName: bucket.bucketWebsiteDomainName,
                protocolPolicy: distribution_1.OriginProtocolPolicy.HTTP_ONLY,
            });
        }
        else {
            return new S3Origin({ domainName: bucket.bucketRegionalDomainName, bucket });
        }
    }
    /**
     * Creates an origin from an HTTP server.
     */
    static fromHttpServer(props) {
        return new HttpOrigin(props);
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     *
     * @internal
     */
    _bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     *
     * @internal
     */
    _renderOrigin() {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            originPath: this.originPath,
            connectionAttempts: this.connectionAttempts,
            connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            originCustomHeaders: this.renderCustomHeaders(),
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props);
        this.bucket = props.bucket;
    }
    /** @internal */
    _bind(scope, options) {
        super._bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(props) {
        var _a, _b;
        super(props);
        this.props = props;
        validateIntInRangeOrUndefined('readTimeout', 1, 60, (_a = props.readTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('keepaliveTimeout', 1, 60, (_b = props.keepaliveTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
    }
    renderCustomOriginConfig() {
        var _a, _b, _c;
        return {
            originProtocolPolicy: (_a = this.props.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: this.props.httpPort,
            httpsPort: this.props.httpsPort,
            originReadTimeout: (_b = this.props.readTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            originKeepaliveTimeout: (_c = this.props.keepaliveTimeout) === null || _c === void 0 ? void 0 : _c.toSeconds(),
        };
    }
}
exports.HttpOrigin = HttpOrigin;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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