import * as acm from "../../aws-certificatemanager";
import { Construct, IResource, Resource, Duration } from "../../core";
import { Origin } from './origin';
/**
 * Interface for CloudFront distributions
 */
export interface IDistribution extends IResource {
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     * @deprecated - Use `distributionDomainName` instead.
     */
    readonly domainName: string;
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     */
    readonly distributionDomainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @attribute
     */
    readonly distributionId: string;
}
/**
 * Attributes used to import a Distribution.
 *
 * @experimental
 */
export interface DistributionAttributes {
    /**
     * The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     */
    readonly domainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @attribute
     */
    readonly distributionId: string;
}
/**
 * Properties for a Distribution
 *
 * @experimental
 */
export interface DistributionProps {
    /**
     * The default behavior for the distribution.
     */
    readonly defaultBehavior: BehaviorOptions;
    /**
     * Additional behaviors for the distribution, mapped by the pathPattern that specifies which requests to apply the behavior to.
     *
     * @default - no additional behaviors are added.
     */
    readonly additionalBehaviors?: Record<string, BehaviorOptions>;
    /**
     * A certificate to associate with the distribution. The certificate must be located in N. Virginia (us-east-1).
     *
     * @default - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
     */
    readonly certificate?: acm.ICertificate;
    /**
     * The price class that corresponds with the maximum price that you want to pay for CloudFront service.
     * If you specify PriceClass_All, CloudFront responds to requests for your objects from all CloudFront edge locations.
     * If you specify a price class other than PriceClass_All, CloudFront serves your objects from the CloudFront edge location
     * that has the lowest latency among the edge locations in your price class.
     *
     * @default PriceClass.PRICE_CLASS_ALL
     */
    readonly priceClass?: PriceClass;
    /**
     * How CloudFront should handle requests that are not successful (e.g., PageNotFound).
     *
     * @default - No custom error responses.
     */
    readonly errorResponses?: ErrorResponse[];
}
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
export declare class Distribution extends Resource implements IDistribution {
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope: Construct, id: string, attrs: DistributionAttributes): IDistribution;
    readonly domainName: string;
    readonly distributionDomainName: string;
    readonly distributionId: string;
    private readonly defaultBehavior;
    private readonly additionalBehaviors;
    private readonly origins;
    private readonly errorResponses;
    private readonly certificate?;
    constructor(scope: Construct, id: string, props: DistributionProps);
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern: string, origin: Origin, behaviorOptions?: AddBehaviorOptions): void;
    private addOrigin;
    private renderOrigins;
    private renderCacheBehaviors;
    private renderErrorResponses;
}
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
export declare enum PriceClass {
    /** USA, Canada, Europe, & Israel */
    PRICE_CLASS_100 = "PriceClass_100",
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PRICE_CLASS_200 = "PriceClass_200",
    /** All locations */
    PRICE_CLASS_ALL = "PriceClass_All"
}
/**
 * How HTTPs should be handled with your distribution.
 */
export declare enum ViewerProtocolPolicy {
    /** HTTPS only */
    HTTPS_ONLY = "https-only",
    /** Will redirect HTTP requests to HTTPS */
    REDIRECT_TO_HTTPS = "redirect-to-https",
    /** Both HTTP and HTTPS supported */
    ALLOW_ALL = "allow-all"
}
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
export declare enum OriginProtocolPolicy {
    /** Connect on HTTP only */
    HTTP_ONLY = "http-only",
    /** Connect with the same protocol as the viewer */
    MATCH_VIEWER = "match-viewer",
    /** Connect on HTTPS only */
    HTTPS_ONLY = "https-only"
}
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
export declare class AllowedMethods {
    /** HEAD and GET */
    static readonly ALLOW_GET_HEAD: AllowedMethods;
    /** HEAD, GET, and OPTIONS */
    static readonly ALLOW_GET_HEAD_OPTIONS: AllowedMethods;
    /** All supported HTTP methods */
    static readonly ALLOW_ALL: AllowedMethods;
    /** HTTP methods supported */
    readonly methods: string[];
    private constructor();
}
/**
 * Options for configuring custom error responses.
 *
 * @experimental
 */
export interface ErrorResponse {
    /**
     * The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ErrorCode.
     *
     * @default - the default caching TTL behavior applies
     */
    readonly ttl?: Duration;
    /**
     * The HTTP status code for which you want to specify a custom error page and/or a caching duration.
     */
    readonly httpStatus: number;
    /**
     * The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.
     *
     * If you specify a value for `responseHttpStatus`, you must also specify a value for `responsePagePath`.
     *
     * @default - not set, the error code will be returned as the response code.
     */
    readonly responseHttpStatus?: number;
    /**
     * The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the
     * `httpStatus`, for example, /4xx-errors/403-forbidden.html
     *
     * @default - the default CloudFront response is shown.
     */
    readonly responsePagePath?: string;
}
/**
 * Options for adding a new behavior to a Distribution.
 *
 * @experimental
 */
export interface AddBehaviorOptions {
    /**
     * HTTP methods to allow for this behavior.
     *
     * @default - GET and HEAD
     */
    readonly allowedMethods?: AllowedMethods;
    /**
     * Whether CloudFront will forward query strings to the origin.
     * If this is set to true, CloudFront will forward all query parameters to the origin, and cache
     * based on all parameters. See `forwardQueryStringCacheKeys` for a way to limit the query parameters
     * CloudFront caches on.
     *
     * @default false
     */
    readonly forwardQueryString?: boolean;
    /**
     * A set of query string parameter names to use for caching if `forwardQueryString` is set to true.
     *
     * @default []
     */
    readonly forwardQueryStringCacheKeys?: string[];
}
/**
 * Options for creating a new behavior.
 *
 * @experimental
 */
export interface BehaviorOptions extends AddBehaviorOptions {
    /**
     * The origin that you want CloudFront to route requests to when they match this behavior.
     */
    readonly origin: Origin;
}
