"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _a;
        this.props = props;
        this.method = (_a = props.method) !== null && _a !== void 0 ? _a : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certifcate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certifcate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certifcate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _a, _b, _c;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of domainNames) {
                const hostedZone = (_b = (_a = validation.props.hostedZones) === null || _a === void 0 ? void 0 : _a[domainName]) !== null && _b !== void 0 ? _b : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_c = validation.props.validationDomains) === null || _c === void 0 ? void 0 : _c[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
//# sourceMappingURL=data:application/json;base64,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