"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlan = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
const rule_1 = require("./rule");
const selection_1 = require("./selection");
const vault_1 = require("./vault");
/**
 * A backup plan
 */
class BackupPlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.rules = [];
        const plan = new backup_generated_1.CfnBackupPlan(this, 'Resource', {
            backupPlan: {
                backupPlanName: props.backupPlanName || id,
                backupPlanRule: core_1.Lazy.anyValue({ produce: () => this.rules }, { omitEmptyArray: true }),
            },
        });
        this.backupPlanId = plan.attrBackupPlanId;
        this.backupPlanArn = plan.attrBackupPlanArn;
        this.versionId = plan.attrVersionId;
        this._backupVault = props.backupVault;
        for (const rule of props.backupPlanRules || []) {
            this.addRule(rule);
        }
    }
    /**
     * Import an existing backup plan
     */
    static fromBackupPlanId(scope, id, backupPlanId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupPlanId = backupPlanId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Daily with 35 day retention
     */
    static daily35DayRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        return plan;
    }
    /**
     * Daily and monthly with 1 year retention
     */
    static dailyMonthly1YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.monthly1Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 5 year retention
     */
    static dailyWeeklyMonthly5YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly5Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 7 year retention
     */
    static dailyWeeklyMonthly7YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly7Year());
        return plan;
    }
    /**
     * Adds a rule to a plan
     *
     * @param rule the rule to add
     */
    addRule(rule) {
        var _a, _b, _c, _d, _e, _f;
        let vault;
        if (rule.props.backupVault) {
            vault = rule.props.backupVault;
        }
        else if (this._backupVault) {
            vault = this._backupVault;
        }
        else {
            this._backupVault = new vault_1.BackupVault(this, 'Vault');
            vault = this._backupVault;
        }
        this.rules.push({
            completionWindowMinutes: (_a = rule.props.completionWindow) === null || _a === void 0 ? void 0 : _a.toMinutes(),
            lifecycle: (rule.props.deleteAfter || rule.props.moveToColdStorageAfter) && {
                deleteAfterDays: (_b = rule.props.deleteAfter) === null || _b === void 0 ? void 0 : _b.toDays(),
                moveToColdStorageAfterDays: (_c = rule.props.moveToColdStorageAfter) === null || _c === void 0 ? void 0 : _c.toDays(),
            },
            ruleName: (_d = rule.props.ruleName) !== null && _d !== void 0 ? _d : `${this.node.id}Rule${this.rules.length}`,
            scheduleExpression: (_e = rule.props.scheduleExpression) === null || _e === void 0 ? void 0 : _e.expressionString,
            startWindowMinutes: (_f = rule.props.startWindow) === null || _f === void 0 ? void 0 : _f.toMinutes(),
            targetBackupVault: vault.backupVaultName,
        });
    }
    /**
     * The backup vault where backups are stored if not defined at
     * the rule level
     */
    get backupVault() {
        if (!this._backupVault) {
            // This cannot happen but is here to make TypeScript happy
            throw new Error('No backup vault!');
        }
        return this._backupVault;
    }
    /**
     * Adds a selection to this plan
     */
    addSelection(id, options) {
        return new selection_1.BackupSelection(this, id, {
            backupPlan: this,
            ...options,
        });
    }
    validate() {
        if (this.rules.length === 0) {
            return ['A backup plan must have at least 1 rule.'];
        }
        return [];
    }
}
exports.BackupPlan = BackupPlan;
//# sourceMappingURL=data:application/json;base64,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