"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.anyValue({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Add command to the startup script of fleet instances.
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _a;
            return ({
                granularity: '1Minute',
                metrics: ((_a = group._metrics) === null || _a === void 0 ? void 0 : _a.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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