"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = void 0;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
//# sourceMappingURL=data:application/json;base64,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