"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
//# sourceMappingURL=data:application/json;base64,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