"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceNamespace = exports.ScalableTarget = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace,
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity,
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    /**
     * Lambda
     */
    ServiceNamespace["LAMBDA"] = "lambda";
    /**
     * Comprehend
     */
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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