"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined && !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod, resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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