"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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