"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * Represents an OpenAPI definition asset.
 * @experimental
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * OpenAPI specification from an S3 archive.
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * OpenAPI specification from an inline JSON object.
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * OpenAPI specification from a local file.
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,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