#! /bin/bash
set -e

# import common functions
src_dir="$(dirname "$0")"
source ${src_dir}/_common.sh

# define polarizations
pol1="xx"
pol2="yy"

# make the file name
bn=$(basename $1 uv)


# We need to run xrfi on the raw visibilities for all polarizations.
# In addition, we want to run on the model visibilities, omnical gain solutions,
# and omnical chi-squared values. We then broadcast all of these together to get a
# consensus of what should be flagged as RFI.
#
# We accomplish this by passing in the *.omni.calfits to the first polarization thread
# (defined by $pol1 above), to flag on the gain solutions and chi-squared values. We pass
# in the model visibilities to the second thread (defined by $pol2). These two threads flag
# these data products in addition to the raw visibitilies. The other two threads only flag
# their respective raw visibilities. The by-product of these steps are *.npz files, which
# contain the flag information produced by each thread.
#
# Once the flagging is done, we need to tie everything together, and broadcast/apply to the
# data files. This is done in the `do_XRFI_APPLY.sh' script, which will take all of the
# *.npz files as arguments in addition to the flags generated by flagging the raw visibilities.

# Define parameters for the xrfi algorithm.
# See documentation in the hera_qm repo for more information about each parameter.
KT_SIZE=8
KF_SIZE=8
SIG_INIT=6
SIG_ADJ=2

if is_same_pol $bn $pol1; then
    # This thread runs on raw visibility + omnical gains/chisquared

    # get the name of the omni.calfits file
    nopol_base=$(remove_pol $bn)
    omni_f=`echo ${nopol_base}HH.uv.omni.calfits`

    # run the xrfi command
    echo xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --calfits_file=${omni_f} --extension=.flags.npz --summary ${bn}HH.uvO
    xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --calfits_file=${omni_f} --extension=.flags.npz --summary ${bn}HH.uvO
elif is_same_pol $bn $pol2; then
    # This thread runs on raw visibility + omnical model visibility
    # get the name of the model visibility file
    nopol_base=$(remove_pol $bn)
    vis_f=`echo ${nopol_base}HH.uv.vis.uvfits`

    # run the xrfi command
    echo xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --model_file=${vis_f} --model_file_format=uvfits --extension=.flags.npz --summary ${bn}HH.uvO
    xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --model_file=${vis_f} --model_file_format=uvfits --extension=.flags.npz --summary ${bn}HH.uvO
else
    # These threads run just on raw visibility
    echo xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --extension=.flags.npz --summary ${bn}HH.uvO
    xrfi_run.py --infile_format=miriad --algorithm=xrfi --kt_size=${KT_SIZE} --kf_size=${KF_SIZE} --sig_init=${SIG_INIT} --sig_adj=${SIG_ADJ} --extension=.flags.npz --summary ${bn}HH.uvO
fi
