"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    /**
     * Grants reading the secret value to some role.
     */
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    /**
     * Grants writing and updating the secret value to some role.
     */
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     */
    get secretValue() {
        return this.secretValueFromJson('');
    }
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     */
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    /**
     * Adds a rotation schedule to the secret.
     */
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Denies the `DeleteSecret` action to all principals within the current account.
     */
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     *
     * Typically, this is just the secret ARN.
     * However, secrets imported by name require a different format.
     */
    get arnForPolicies() { return this.secretArn; }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        /**
         *
         */
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        this.secretName = this.physicalName;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principal);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    /**
     *
     */
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Imports a secret by secret name;
     *
     * the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     */
    static fromSecretName(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    sep: ':',
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.secretName = parseSecretName(scope, attrs.secretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (deprecated) Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         *
         */
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    /**
     *
     */
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).parseArn(secretArn, ':').resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${SecretsManager suffix}`
        // If there is no hyphen, assume no suffix was provided, and return the whole name.
        return resourceName.substr(0, resourceName.lastIndexOf('-')) || resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
//# sourceMappingURL=data:application/json;base64,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