"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const tmp = require("tmp");
const fs = require("fs");
class Seeder extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.setup || !Array.isArray(props.setup))
            throw new Error('setup value must be an array of JSON objects');
        this.props = props;
        const destinationBucket = new aws_s3_1.Bucket(this, 'acds-bucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            this.writeTempFile(dir, 'setup.json', props.setup);
            if (props.teardown) {
                this.writeTempFile(dir, 'teardown.json', props.teardown);
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false,
            });
        });
        const fn = new aws_lambda_1.Function(this, 'handler', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.fromInline(`
console.log('function loaded');

const AWS = require('aws-sdk');
const s3 = new AWS.S3();

const writeTypeFromAction = (action) => {
  if (action === "put")
    return "Item";
  if (action === "delete")
    return "Key";
}

const run = async (filename, action) => {
  console.log('reading from s3');
  const data = await s3.getObject({
    Bucket: "${destinationBucket.bucketName}", 
    Key: filename
  }).promise();
  console.log('finished reading from s3');
  
  console.log('transforming seed data');
  const seed = JSON.parse(data.Body.toString());
  console.log('finished transforming seed data');
  
  const documentClient = new AWS.DynamoDB.DocumentClient({
    convertEmptyValues: true
  });
  console.log('sending data to dynamodb');
  for(let i = 0; i < seed.length;i++) {
    await documentClient[action]({
      TableName: '${props.table.tableName}',
      [writeTypeFromAction(action)]: seed[i]
    }).promise();
  };
  console.log('finished sending data to dynamodb');
}

exports.handler = async (event) => {
  if (event.mode === "delete" || event.mode === "update")
    await run("teardown.json", "delete");
  if (event.mode === "create" || event.mode === "update")
    await run("setup.json", "put");
}`),
        });
        destinationBucket.grantRead(fn);
        props.table.grantWriteData(fn);
        const onEvent = new custom_resources_1.AwsCustomResource(this, 'on-event', {
            onCreate: {
                ...this.callLambdaOptions(),
                parameters: {
                    FunctionName: fn.functionArn,
                    InvokeArgs: JSON.stringify({
                        mode: 'create',
                    }),
                },
            },
            onDelete: props.teardown
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'delete',
                        }),
                    },
                }
                : undefined,
            onUpdate: props.refreshOnUpdate
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'update',
                        }),
                    },
                }
                : undefined,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        fn.grantInvoke(onEvent);
    }
    callLambdaOptions() {
        return {
            service: 'Lambda',
            action: 'invokeAsync',
            apiVersion: '2015-03-31',
            physicalResourceId: {
                id: `${this.props.table.tableArn}-seeder`,
            },
        };
    }
    writeTempFile(dir, filename, data) {
        const buffer = Buffer.from(JSON.stringify(data));
        const filepath = dir + '/' + filename;
        fs.writeFileSync(filepath, buffer);
    }
}
exports.Seeder = Seeder;
//# sourceMappingURL=data:application/json;base64,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