# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['restful_aws_lambda']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'restful-aws-lambda',
    'version': '0.1.2',
    'description': 'Lightweight framework for building REST API using AWS Lambda + API Gateway',
    'long_description': '# restful-aws-lambda\n\n[![CI/CD Status](https://github.com/joffreybvn/restful-aws-lambda/workflows/CI%2FCD/badge.svg?branch=master)](https://github.com/joffreybvn/restful-aws-lambda/actions?query=branch:master)\n[![Coverage Status](https://coveralls.io/repos/github/Joffreybvn/restful-aws-lambda/badge.svg?branch=master)](https://coveralls.io/github/Joffreybvn/restful-aws-lambda?branch=master)\n[![Latest Version](https://img.shields.io/pypi/v/restful-aws-lambda.svg?color=blue)](https://pypi.python.org/pypi/restful-aws-lambda)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/restful-aws-lambda?label=pypi%20downloads)](https://pypi.org/project/restful-aws-lambda/)\n![License](https://img.shields.io/github/license/joffreybvn/restful-aws-lambda)\n\nrestful-aws-lambda is a lightweight opinionated framework for building REST API using [AWS Lambda](https://aws.amazon.com/lambda/) and [API Gateway](https://aws.amazon.com/api-gateway/).\n\n## Motivation\n\nWhy another framework ?\n\nWhen using API Gateway and python Lambda functions, the most common pattern is to have a unique Lambda function triggered by a proxy API Gateway resource. The Lambda then uses a framework like [Flask](https://flask.palletsprojects.com/en/1.1.x/) to do all the routing. In an API Gateway + Lambda context, I feel like **the routing should be handled by API Gateway itself**, then forwarding the request to specific Lambda functions for each resource or endpoint.\n\n## Features\n\n- No routing. Yes, this is a feature. Routing should be handled by API Gateway.\n- API Gateway event parsing (including request body and path parameters).\n- Cleaner syntax.\n- Customizable JSON dumps behavior\n- No schema validation\n\n## Installation\n\nInstall the package from PyPI using pip:\n\n```\n$ pip install restful-aws-lambda\n```\n\nrestful-aws-lambda should also be included in the deployment package of your Lambda functions.\n\n## Getting started\n\nrestful-aws-lambda provides a `@route` decorator to parse the API Gateway event into a `Request` object available in the handler function as an argument. It also formats the handler\'s output to the expected Lambda + API Gateway format seamlessly.\n\nTurning this:\n\n```python\nimport json\n\ndef handler(event, context):\n    body = json.loads(event["body"])\n    query_params = event["queryStringParameters"]\n    path_params = event["pathParameters"]\n\n    return {\n        "statusCode": 200,\n        "body": json.dumps({\n            "message": f"Hello from AWS Lambda {body[\'name\']}!!"\n        })\n    }\n\n```\n\nInto this:\n\n```python\nfrom restful_aws_lambda import route\n\n\n@route\ndef handler(request):\n    body = request.json\n    query_params = request.query_params\n    path_params = request.path_params\n\n    return 200, {"message": f"Hello from AWS Lambda {body[\'name\']}!!"}\n```\n\nYou can still access the original `event` and `context` arguments from the handler:\n\n```python\nfrom restful_aws_lambda import route\n\n\n@route\ndef handler(request, event, context):\n    print(event)\n    body = request.json\n\n    return 200, {"message": f"Hello from AWS Lambda {body[\'name\']}!!"}\n```\n\n<br/>\n\nPath parameters defined in API Gateway can also be accessed directly as function argument:\n\n<br/>\n\n![api-gateway-path-params](https://raw.githubusercontent.com/joffreybvn/restful-aws-lambda/master/images/api-gateway-path-params.png)\n\n```python\nfrom restful_aws_lambda import route\n\n\n@route\ndef get_user(user_id):\n    print(user_id)\n\n    # get user from db\n    user = {"id": user_id, "name": "John Doe"}\n\n    return 200, user\n```\n\n## JSON dumps options\n\nrestful-aws-lambda uses the default `json` library to dump the lambda handler response. You can customize the behavior of it by passing `json.dumps()` keyword arguments to the `json=` parameter:\n\n```python\nimport json\nimport datetime\nfrom restful_aws_lambda import route\n\nclass JSONDatetimeEncoder(json.JSONEncoder):\n    def default(self, obj):\n        if isinstance(obj, (datetime.date, datetime.datetime)):\n            return obj.isoformat()\n\n        return super(JSONDatetimeEncoder, self).default(obj)\n\n\n@route(json={"cls": JSONDatetimeEncoder})\ndef lambda_handler(request):\n    today = datetime.date.today()\n    return 200, {"today": today}\n```\n\n## Example\n\nYou can look at the [sample](https://github.com/joffreybvn/restful-aws-lambda/tree/master/sample) for a minimal pylambdarest API.\n\nIn this sample, we use the [serverless](https://www.serverless.com/) framework to declare the API Gateway -> Lambda routing\n\nThe packaging of the Lambda functions is done using the [serverless-python-requirements](https://github.com/UnitedIncome/serverless-python-requirements) plugin.\n',
    'author': 'Joffrey Bienvenu',
    'author_email': 'joffreybvn@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/Joffreybvn/restful-aws-lambda',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
