"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveNewProject = exports.renderProjenNewOptions = void 0;
const inventory = require("../inventory");
const option_hints_1 = require("../option-hints");
const PROJEN_NEW = '__new__';
const TAB = makePadding(2);
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenNewOptions(fqn, args, comments = option_hints_1.NewProjectOptionHints.NONE) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args, comments },
    };
}
exports.renderProjenNewOptions = renderProjenNewOptions;
function resolveNewProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
        comments: f.comments,
    };
}
exports.resolveNewProject = resolveNewProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c;
    const renders = {};
    const optionsWithDefaults = [];
    const useSingleQuotes = (str) => str === null || str === void 0 ? void 0 : str.replace(/"(.+)"/, '\'$1\'');
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        if (opts.args[optionName] !== undefined) {
            const value = opts.args[optionName];
            const js = JSON.stringify(value);
            renders[optionName] = `${optionName}: ${useSingleQuotes(js)},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            renders[optionName] = `// ${optionName}: ${useSingleQuotes(defaultValue)},`;
        }
    }
    const bootstrap = (_c = opts.bootstrap) !== null && _c !== void 0 ? _c : false;
    if (bootstrap) {
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({ args: opts.args, fqn: opts.type.fqn, comments: opts.comments })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // generate rendering
    const result = [];
    result.push('{');
    // render options with defaults
    optionsWithDefaults.sort();
    for (const optionName of optionsWithDefaults) {
        result.push(`${TAB}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults as comments
    if (opts.comments === option_hints_1.NewProjectOptionHints.ALL) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined);
        result.push(...renderCommentedOptionsByModule(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.FEATURED) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined && opt.featured);
        result.push(...renderCommentedOptionsInOrder(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.NONE) {
        // don't render any extra options
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return result.join('\n');
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function renderCommentedOptionsByModule(renders, options) {
    var _a;
    const optionsByModule = {};
    for (const option of options) {
        const parentModule = option.parent;
        optionsByModule[parentModule] = (_a = optionsByModule[parentModule]) !== null && _a !== void 0 ? _a : [];
        optionsByModule[parentModule].push(option);
    }
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const [moduleName, optionGroup] of Object.entries(optionsByModule).sort()) {
        result.push(`${TAB}/* ${moduleName} */`);
        for (const option of optionGroup) {
            const paramRender = renders[option.name];
            const docstring = option.docs || 'No documentation found.';
            result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
        }
        result.push('');
    }
    return result;
}
function renderCommentedOptionsInOrder(renders, options) {
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const option of options) {
        const paramRender = renders[option.name];
        const docstring = option.docs || 'No documentation found.';
        result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
    }
    return result;
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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