"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const yaml = require("yaml");
const __1 = require("..");
const deps_1 = require("../deps");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        expect(npm.npmTokenSecret).toStrictEqual('NPM_TOKEN');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
            npmTokenSecret: 'GITHUB_TOKEN',
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmTokenSecret).toStrictEqual('GITHUB_TOKEN');
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.releaseWorkflow) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            'runs-on': 'ubuntu-latest',
            'env': {
                CI: 'true',
            },
            'steps': [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: true,
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
describe('scripts', () => {
    test('removeScript will remove tasks and scripts', () => {
        const p = new TestNodeProject();
        p.addTask('chortle', { exec: 'echo "frabjous day!"' });
        p.setScript('slithy-toves', 'gyre && gimble');
        expect(packageJson(p).scripts).toHaveProperty('chortle');
        expect(packageJson(p).scripts).toHaveProperty('slithy-toves');
        p.removeScript('chortle');
        p.removeScript('slithy-toves');
        expect(packageJson(p).scripts).not.toHaveProperty('chortle');
        expect(packageJson(p).scripts).not.toHaveProperty('slithy-toves');
    });
});
test('buildWorkflowMutable will push changes to PR branches', () => {
    // WHEN
    const project = new TestNodeProject({
        mutableBuild: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
});
test('projenDuringBuild can be used to disable "projen synth" during build', () => {
    const enabled = new TestNodeProject({
        projenDuringBuild: true,
    });
    const disabled = new TestNodeProject({
        projenDuringBuild: false,
    });
    const buildTaskEnabled = util_1.synthSnapshot(enabled)['.projen/tasks.json'].tasks.build;
    const buildTaskDisabled = util_1.synthSnapshot(disabled)['.projen/tasks.json'].tasks.build;
    expect(buildTaskEnabled.steps[0].exec).toEqual('npx projen');
    expect(buildTaskDisabled.steps).toBeUndefined();
});
test('projen synth is only executed for subprojects', () => {
    // GIVEN
    const root = new TestNodeProject();
    // WHEN
    new TestNodeProject({ parent: root, outdir: 'child' });
    // THEN
    const snapshot = util_1.synthSnapshot(root);
    const rootBuildTask = snapshot['.projen/tasks.json'].tasks.build;
    const childBuildTask = snapshot['child/.projen/tasks.json'].tasks.build;
    expect(rootBuildTask).toStrictEqual({
        category: '00.build',
        description: 'Full release build (test+compile)',
        name: 'build',
        steps: [{ exec: 'npx projen' }],
    });
    expect(childBuildTask).toStrictEqual({
        category: '00.build',
        description: 'Full release build (test+compile)',
        name: 'build',
    });
});
test('enabling dependabot does not overturn mergify: false', () => {
    // WHEN
    const project = new TestNodeProject({
        dependabot: true,
        mergify: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    // Note: brackets important, they prevent "." in filenames to be interpreted
    //       as JSON object path delimiters.
    expect(snapshot).not.toHaveProperty(['.mergify.yml']);
    expect(snapshot).toHaveProperty(['.github/dependabot.yml']);
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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