/*
 * Decompiled with CFR 0.152.
 */
package nl.esciencecenter.xenon.adaptors.filesystems.sftp;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import nl.esciencecenter.xenon.XenonException;
import nl.esciencecenter.xenon.adaptors.NotConnectedException;
import nl.esciencecenter.xenon.adaptors.XenonProperties;
import nl.esciencecenter.xenon.adaptors.filesystems.EndOfFileException;
import nl.esciencecenter.xenon.adaptors.filesystems.NoSpaceException;
import nl.esciencecenter.xenon.adaptors.filesystems.PathAttributesImplementation;
import nl.esciencecenter.xenon.adaptors.filesystems.PermissionDeniedException;
import nl.esciencecenter.xenon.adaptors.filesystems.PosixFileUtils;
import nl.esciencecenter.xenon.filesystems.DirectoryNotEmptyException;
import nl.esciencecenter.xenon.filesystems.FileSystem;
import nl.esciencecenter.xenon.filesystems.InvalidPathException;
import nl.esciencecenter.xenon.filesystems.NoSuchPathException;
import nl.esciencecenter.xenon.filesystems.Path;
import nl.esciencecenter.xenon.filesystems.PathAlreadyExistsException;
import nl.esciencecenter.xenon.filesystems.PathAttributes;
import nl.esciencecenter.xenon.filesystems.PosixFilePermission;
import org.apache.sshd.client.subsystem.sftp.SftpClient;
import org.apache.sshd.common.subsystem.sftp.SftpException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SftpFileSystem
extends FileSystem {
    private static final Logger LOGGER = LoggerFactory.getLogger(SftpFileSystem.class);
    private final SftpClient client;

    protected SftpFileSystem(String uniqueID, String name, String location, Path entryPath, int bufferSize, SftpClient client, XenonProperties properties) {
        super(uniqueID, name, location, entryPath, bufferSize, properties);
        this.client = client;
    }

    @Override
    public void close() throws XenonException {
        LOGGER.debug("close fileSystem = {}", (Object)this);
        try {
            this.client.close();
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to close sftp client");
        }
        super.close();
        LOGGER.debug("close OK");
    }

    @Override
    public boolean isOpen() throws XenonException {
        return this.client.isOpen();
    }

    @Override
    public void rename(Path source, Path target) throws XenonException {
        LOGGER.debug("move source = {} target = {}", (Object)source, (Object)target);
        Path absSource = this.toAbsolutePath(source);
        Path absTarget = this.toAbsolutePath(target);
        this.assertPathExists(absSource);
        if (this.areSamePaths(absSource, absTarget)) {
            return;
        }
        this.assertPathNotExists(absTarget);
        this.assertParentDirectoryExists(absTarget);
        try {
            this.client.rename(absSource.toString(), absTarget.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to rename path");
        }
        LOGGER.debug("move OK");
    }

    @Override
    public void createDirectory(Path dir) throws XenonException {
        LOGGER.debug("createDirectory dir = {}", (Object)dir);
        Path absDir = this.toAbsolutePath(dir);
        this.assertPathNotExists(absDir);
        this.assertParentDirectoryExists(absDir);
        try {
            this.client.mkdir(absDir.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to mkdir");
        }
        LOGGER.debug("createDirectory OK");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void createFile(Path file) throws XenonException {
        Path absFile = this.toAbsolutePath(file);
        this.assertPathNotExists(absFile);
        LOGGER.debug("createFile path = {}", (Object)absFile);
        OutputStream out = null;
        try {
            out = this.writeToFile(absFile);
        }
        finally {
            try {
                if (out != null) {
                    out.close();
                }
            }
            catch (IOException e) {}
        }
        LOGGER.debug("createFile OK");
    }

    @Override
    public void createSymbolicLink(Path link, Path path) throws XenonException {
        Path absLink = this.toAbsolutePath(link);
        this.assertPathNotExists(absLink);
        this.assertParentDirectoryExists(absLink);
        try {
            this.client.symLink(absLink.toString(), path.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Cannot create link: " + absLink + " -> " + path);
        }
    }

    @Override
    protected void deleteFile(Path file) throws XenonException {
        try {
            this.client.remove(file.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Cannot delete file: " + file);
        }
    }

    @Override
    protected void deleteDirectory(Path dir) throws XenonException {
        try {
            this.client.rmdir(dir.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Cannot delete directory: " + dir);
        }
    }

    private SftpClient.Attributes stat(Path path) throws XenonException {
        SftpClient.Attributes result;
        LOGGER.debug("* stat path = {}", (Object)path);
        try {
            result = this.client.lstat(path.toString());
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to retrieve attributes from: " + path);
        }
        LOGGER.debug("* stat OK result = {}", (Object)result);
        return result;
    }

    @Override
    public boolean exists(Path path) throws XenonException {
        LOGGER.debug("exists path = {}", (Object)path);
        try {
            this.stat(this.toAbsolutePath(path));
            return true;
        }
        catch (NoSuchPathException e) {
            return false;
        }
    }

    protected List<PathAttributes> listDirectory(Path path) throws XenonException {
        try {
            this.assertDirectoryExists(path);
            ArrayList<PathAttributes> result = new ArrayList<PathAttributes>();
            for (SftpClient.DirEntry f : this.client.readDir(path.toString())) {
                result.add(SftpFileSystem.convertAttributes(path.resolve(f.getFilename()), f.getAttributes()));
            }
            return result;
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to list directory " + path);
        }
    }

    @Override
    public InputStream readFromFile(Path path) throws XenonException {
        InputStream in;
        LOGGER.debug("newInputStream path = {}", (Object)path);
        Path absPath = this.toAbsolutePath(path);
        this.assertFileExists(absPath);
        try {
            in = this.client.read(absPath.toString());
        }
        catch (IOException e) {
            throw new XenonException("sftp", "Failed to open stream to read from " + absPath, e);
        }
        LOGGER.debug("newInputStream OK");
        return in;
    }

    @Override
    public OutputStream writeToFile(Path path, long size) throws XenonException {
        Path absPath = this.toAbsolutePath(path);
        this.assertPathNotExists(absPath);
        this.assertParentDirectoryExists(absPath);
        try {
            return this.client.write(absPath.toString(), new SftpClient.OpenMode[]{SftpClient.OpenMode.Write, SftpClient.OpenMode.Create, SftpClient.OpenMode.Truncate});
        }
        catch (IOException e) {
            throw new XenonException("sftp", "Failed open stream to write to: " + absPath, e);
        }
    }

    @Override
    public OutputStream writeToFile(Path path) throws XenonException {
        return this.writeToFile(path, -1L);
    }

    @Override
    public OutputStream appendToFile(Path path) throws XenonException {
        Path absPath = this.toAbsolutePath(path);
        this.assertFileExists(absPath);
        try {
            return this.client.write(absPath.toString(), new SftpClient.OpenMode[]{SftpClient.OpenMode.Write, SftpClient.OpenMode.Append});
        }
        catch (IOException e) {
            throw new XenonException("sftp", "Failed open stream to write to: " + absPath, e);
        }
    }

    @Override
    public PathAttributes getAttributes(Path path) throws XenonException {
        Path absPath = this.toAbsolutePath(path);
        return SftpFileSystem.convertAttributes(absPath, this.stat(absPath));
    }

    @Override
    public Path readSymbolicLink(Path link) throws XenonException {
        Path result;
        LOGGER.debug("readSymbolicLink path = {}", (Object)link);
        Path absLink = this.toAbsolutePath(link);
        this.assertFileIsSymbolicLink(absLink);
        try {
            String target = this.client.readLink(absLink.toString());
            if (!target.startsWith(File.separator)) {
                Path parent = absLink.getParent();
                result = parent.resolve(target);
            } else {
                result = new Path(target);
            }
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to read link: " + absLink);
        }
        LOGGER.debug("readSymbolicLink OK result = {}", (Object)result);
        return result;
    }

    @Override
    public void setPosixFilePermissions(Path path, Set<PosixFilePermission> permissions) throws XenonException {
        LOGGER.debug("setPosixFilePermissions path = {} permissions = {}", (Object)path, permissions);
        if (permissions == null) {
            throw new IllegalArgumentException("Permissions is null");
        }
        Path absPath = this.toAbsolutePath(path);
        this.assertPathExists(absPath);
        try {
            SftpClient.Attributes a = new SftpClient.Attributes();
            a.setPermissions(PosixFileUtils.permissionsToBits(permissions));
            this.client.setStat(absPath.toString(), a);
        }
        catch (IOException e) {
            throw SftpFileSystem.sftpExceptionToXenonException(e, "Failed to set permissions on: " + absPath);
        }
        LOGGER.debug("setPosixFilePermissions OK");
    }

    private static long convertTime(FileTime time) {
        return time.toMillis();
    }

    private static PathAttributes convertAttributes(Path path, SftpClient.Attributes attributes) {
        PathAttributesImplementation result = new PathAttributesImplementation();
        result.setPath(path);
        result.setDirectory(attributes.isDirectory());
        result.setRegular(attributes.isRegularFile());
        result.setOther(attributes.isOther());
        result.setSymbolicLink(attributes.isSymbolicLink());
        if (attributes.getModifyTime() == null) {
            result.setLastModifiedTime(0L);
        } else {
            result.setLastModifiedTime(SftpFileSystem.convertTime(attributes.getModifyTime()));
        }
        if (attributes.getCreateTime() == null) {
            result.setCreationTime(result.getLastModifiedTime());
        } else {
            result.setCreationTime(SftpFileSystem.convertTime(attributes.getCreateTime()));
        }
        if (attributes.getAccessTime() == null) {
            result.setCreationTime(result.getLastModifiedTime());
        } else {
            result.setLastAccessTime(SftpFileSystem.convertTime(attributes.getAccessTime()));
        }
        result.setSize(attributes.getSize());
        Set<PosixFilePermission> permission = PosixFileUtils.bitsToPermissions(attributes.getPermissions());
        result.setPermissions(permission);
        result.setExecutable(permission.contains((Object)PosixFilePermission.OWNER_EXECUTE));
        result.setReadable(permission.contains((Object)PosixFilePermission.OWNER_READ));
        result.setWritable(permission.contains((Object)PosixFilePermission.OWNER_WRITE));
        result.setGroup(attributes.getGroup());
        result.setOwner(attributes.getOwner());
        result.setHidden(path.getFileNameAsString().startsWith("."));
        return result;
    }

    static XenonException sftpExceptionToXenonException(IOException e, String message) {
        if (e instanceof SftpException) {
            SftpException x = (SftpException)e;
            switch (x.getStatus()) {
                case 1: {
                    return new EndOfFileException("sftp", "Unexpected EOF", e);
                }
                case 2: 
                case 10: {
                    return new NoSuchPathException("sftp", "Path does not exists", e);
                }
                case 3: {
                    return new PermissionDeniedException("sftp", "Permission denied", e);
                }
                case 6: {
                    return new NotConnectedException("sftp", "Not connected", e);
                }
                case 7: {
                    return new NotConnectedException("sftp", "Connection lost", e);
                }
                case 8: {
                    return new XenonException("sftp", "Unsupported operation", e);
                }
                case 11: {
                    return new PathAlreadyExistsException("sftp", "Already exists", e);
                }
                case 12: {
                    return new PermissionDeniedException("sftp", "Write protected", e);
                }
                case 22: {
                    return new PermissionDeniedException("sftp", "Cannot delete", e);
                }
                case 27: {
                    return new PermissionDeniedException("sftp", "Delete pending", e);
                }
                case 13: 
                case 14: {
                    return new NoSpaceException("sftp", "No space on filesystem", e);
                }
                case 15: {
                    return new NoSpaceException("sftp", "Quota exceeded", e);
                }
                case 28: {
                    return new InvalidPathException("sftp", "File corrupt", e);
                }
                case 18: {
                    return new DirectoryNotEmptyException("sftp", "Directory not empty", e);
                }
                case 19: {
                    return new InvalidPathException("sftp", "Not a directory", e);
                }
                case 20: {
                    return new InvalidPathException("sftp", "Invalid file name", e);
                }
                case 21: {
                    return new InvalidPathException("sftp", "Link loop", e);
                }
                case 24: {
                    return new InvalidPathException("sftp", "File is a directory", e);
                }
                case 29: {
                    return new XenonException("sftp", "Invalid owner", e);
                }
                case 30: {
                    return new XenonException("sftp", "Invalid group", e);
                }
                case 9: {
                    return new XenonException("sftp", "Invalid handle", e);
                }
                case 23: {
                    return new XenonException("sftp", "Invalid parameter", e);
                }
                case 17: 
                case 25: 
                case 26: 
                case 31: {
                    return new XenonException("sftp", "Locking failed", e);
                }
                case 16: {
                    return new XenonException("sftp", "Unknown principal", e);
                }
                case 5: {
                    return new XenonException("sftp", "Malformed message", e);
                }
            }
        }
        if (e.getMessage().contains("client is close")) {
            return new NotConnectedException("sftp", e.getMessage());
        }
        return new XenonException("sftp", message, e);
    }
}

