# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['konverter']

package_data = \
{'': ['*']}

install_requires = \
['click>=7.0,<8.0',
 'cryptography>=2.8,<3.0',
 'jinja2>=2.10.3,<3.0.0',
 'ruamel.yaml>=0.16.5,<0.17.0']

entry_points = \
{'console_scripts': ['konverter = konverter.__main__:cli',
                     'konverter-vault = konverter.vault:cli']}

setup_kwargs = {
    'name': 'konverter',
    'version': '0.3.0',
    'description': 'YAML templating for responsible users',
    'long_description': '# Konverter - YAML templating for responsible users\n\nKonverter is a YAML templating tool that uses Jinja [expression](https://jinja.palletsprojects.com/en/2.10.x/templates/#expressions) and [templates](https://jinja.palletsprojects.com/en/2.10.x/templates/) and supports inline-encrypted variables.\n\n**Features:**\n- generic YAML templating (**not** specific to e.g. Kubernetes)\n- works on the **YAML structure**; text templating only available for values\n- support for simple **inline-encrypted variables**\n- ... more to come (see "[planned features](#planned-features)")\n\nThe project is still at an **early stage**, so I\'m happy about any contribution!\n\n### Motivation\n\nI created this tool mainly to scratch my own itch. I needed a simple way to store encrypted values along my Kubernetes manifests, but found that e.g. [Kustomize](https://kustomize.io/) makes this [extra hard](https://github.com/kubernetes-sigs/kustomize/issues/692).\n\nIn that regard Konverter follows the Python philosophy: "We are all [consenting adults](https://docs.python-guide.org/writing/style/#we-are-all-responsible-users)". So yes, you can do potentially dangerous things with it.\n\nAlso, the way Kustomize works by [patching and transforming](https://github.com/kubernetes-sigs/kustomize/blob/master/docs/fields.md#transformers) YAML, as well as the concept of `Secret` and `ConfigMap` [generators](https://github.com/kubernetes-sigs/kustomize/blob/master/docs/fields.md#generators), didn\'t feel natural to me.\n\nTo each his own :)\n\n## Installation\n\nThe easiest way to install Konverter is via [pipx](https://github.com/pipxproject/pipx):\n\n```shell\n$ pipx install konverter\n```\n\nOr just with plain pip:\n\n```shell\n$ pip install konverter\n```\n\n## Usage\n\nKonverter is configured via a YAML file (of course). In this config file, you define your templates and the context (variables) used to render those templates. If you want to render your templates with a different context (e.g. templating Kubernetes manifests for staging, production, ...) you need a separate config file.\n\nHere\'s a simple example of a config file:\n```yaml\ntemplates:\n  - manifest.yaml\n\ncontext:\n  - vars/common.yaml\n  - vars/prod.yaml\n```\n\nThe templates are rendered using the `konverter` command by passing it the config file:\n```shell\n$ konverter production.yaml\n...\n```\n\nThe rendered templates will be written to `stdout` as a multi-document YAML. This allows for easy composition with other tools like `kubectl`:\n\n```shell\n$ konverter production.yaml | kubectl apply -f -\n```\n\n### Templates\n\nUnder the `templates` key, you can configure a list of files or directories. In the case of a directory, Konverter will recursively include all `*.y[a]ml` files.\n\nKonverter uses [YAML tags](https://yaml.org/spec/1.2/spec.html#id2761292) to declare expressions (`!k/expr`) and templates (`!k/template`):\n\n```yaml\nfoo: !k/expr foo.bar\nFOO: !k/expr foo.bar|upper\n\nnot_defined: !k/expr missing_variable|default(\'default value\')\n\nconfig: !k/template |-\n  Hello {{ name }}\n\n  {% for item in list_of_values %}\n    Hello {{ item }}\n  {% endfor %}\n```\n\n#### Expressions\n\nExpressions are mainly useful for injecting data from the context and performing simple transformations:\n\n```yaml\n# Example context\nvalue:\n  from:\n    context: barfoo\n    other: foobar\n```\n\n```yaml\n# Use !k/expr to inject the values in a template\nfoo:\n  bar: !k/expr value.from.context\n  foo: !k/expr value.from.other|upper\n```\n\nWhen using the `!k/expr` tag, the value can be any valid [Jinja2 expression](https://jinja.palletsprojects.com/en/2.10.x/templates/#expressions). In addition to the standard Jinja [filters](https://jinja.palletsprojects.com/en/2.10.x/templates/#list-of-builtin-filters), [tests](https://jinja.palletsprojects.com/en/2.10.x/templates/#list-of-builtin-tests) and [global functions](https://jinja.palletsprojects.com/en/2.10.x/templates/#list-of-global-functions) the following filters are currently available:\n\n- `to_json`: dump an object as a *JSON string*\n- `b64encode`: encode value using Base64\n\nMissing a filter? Let me know or feel free to contribute!\n\n#### Templates\n\nThe `!k/template` tag supports the full [Jinja2 template syntax](https://jinja.palletsprojects.com/en/2.10.x/templates/) (currently with some exceptions, like template inheritance), but the output will always be a string value. Inline templates are most useful for customizing config files.\n\n### Context\n\nThe `context` key in the config contains a list of files or directories that Konverter uses for loading variable definitions. Those variables serve as the context (hence the name) when rendering templates. In the case of multiple context files containing the same top-level key, the entry from the last listed file wins (in the future we might support merging those contexts).\n\nThis is how a simple context file might look like\n```yaml\ndeployment:\n  name: foobar\n  server_name: foobar.example.com\nannotations:\n  version: 1.0\n  environment: production\n```\n\n#### Encrypted values\n\nIn addition there can be inline-encrypted values:\n\n```yaml\ncredentials:\n  user: root\n  password: !k/vault gAAAAABeKd7EEt-jCYJSLS_ze6oO401yRDCthJFkuR4ptIFNnSElTccUnOVSQ1rSCDbIdljB59SRWjy2rDq7174stq3FFzyE_w==\n```\n\nKonverter uses [Fernet symmetric encryption](https://cryptography.io/en/latest/fernet/) with secret keys. To use encrypted values we have to create key first:\n\n```shell\n$ konverter-vault keygen\n```\n\nThis will create a key file `.konverter-vault` in the current directory containing the key. **Make sure to never commit this file to version control!**\n\nBy default, Konverter will look for the vault key in the same directory as the config file when rendering templates. See the section "[advanced configuration](#advanced-configuration)" on how to use a different path.\n\nThe following command will decrypt any encrypted values in the given file and launch your `$EDITOR`:\n\n```shell\n$ konverter-vault edit vars/secret.yaml\n```\n\nValues that should be encrypted are marked with the `!k/encrypt` YAML tag:\n\n```yaml\ncredentials:\n  user: root\n  password: !k/encrypt secret-password\n```\n\nAfter closing the editor Konverter will encrypt the values:\n\n```yaml\ncredentials:\n  user: root\n  password: !k/vault gAAAAABeKd7EEt-jCYJSLS_ze6oO401yRDCthJFkuR4ptIFNnSElTccUnOVSQ1rSCDbIdljB59SRWjy2rDq7174stq3FFzyE_w==\n```\n\nInstead of editing the file via the `konverter-vault edit` command you can also encrypt/decrypt the files separately:\n\n```shell\n$ konverter-vault encrypt vars/secret.yaml\n...\n$ konverter-vault decrypt vars/secret.yaml\n```\n\nIn case the decrypted content should be passed to another tool that expects\nYAML or JSON, the `konverter-vault show` command can come in handy:\n\n```shell\n$ konverter-vault show --format=json vars/secret.yaml\n{\n  "credentials": {\n    "user": "root",\n    "password": "secret-password"\n  }\n}\n```\n\nIt will decrypt the given file and remove all `!k/(encrypt|vault)` YAML tags.\nSupported output formats are `yaml` (default), `json` or `terraform`.\n\nThe `terraform` output format is usefull for using `konverter-vault` as an\n[external data source in\nTerraform](https://www.terraform.io/docs/providers/external/data_source.html)\n\n```hcl\ndata "external" "konverter" {\n  program = [\n    "konverter-vault", "show", "--format=terraform", "vars/secrets.yaml"\n  ]\n}\n```\n\nUnfortunately the "[external program\nprotocol](https://www.terraform.io/docs/providers/external/data_source.html#external-program-protocol)"\nonly allows string values (no lists or objects). All `float`, `int` and `bool`\nvalues will be converted to strings. Other types will cause an error when\nusing this output format.\n\n### Advanced configuration\n\nThe above config file could be rewritten in a more verbose format:\n\n```yaml\ntemplates:\n  - manifest.yaml\n\nproviders:\n  default:\n    key_path: .konverter-vault\n\ncontext:\n  - provider: default\n    path: vars/common.yaml\n  - provider: default\n    path: vars/prod.yaml\n```\n\nIn case we want to change the location of the vault key, we can simply override the default provider:\n\n```yaml\n...\nproviders:\n  default:\n    key_path: ~/.my-vault-key\n...\n```\n\n## Planned features\n\n- better error handling\n- improve test coverage\n- importing templates from a file\n- support for other context providers\n\nFeel free to create an issue or a pull-request if you are missing something!\n',
    'author': 'Simon Westphahl',
    'author_email': 'simon@westphahl.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/westphahl/konverter',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
