# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lethargy']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'lethargy',
    'version': '2.0.1',
    'description': 'A minimal library to make your option-parsing easier.',
    'long_description': '# Lethargy — Declarative option parsing, for developers\n\n[![Released version (shield)]][Version URL]\n[![Size (shield)]][Size URL]\n\n[Released version (shield)]: https://img.shields.io/pypi/v/lethargy?color=blue\n[Version URL]: https://pypi.org/project/lethargy\n\n[Size (shield)]: https://img.shields.io/badge/size-14%20kB-blue\n[Size URL]: https://github.com/SeparateRecords/lethargy/tree/master/lethargy\n<!-- Size correct as at e4db57f (March 16, 2020) -->\n\nLethargy takes care of option parsing in your scripts, so you can be more productive when writing the important stuff. It\'s simple, concise, explicit, and Pythonic.\n\nUnlike [Click](https://click.palletsprojects.com/en/7.x/) and [Argparse](https://docs.python.org/3/library/argparse.html), Lethargy is succinct, can be implemented without changing the structure of a program, and requires no boilerplate code. This makes it especially suited to scripting and prototyping.\n\nBy design, it is not a full argument parser. If you\'re building a complete CLI application, you\'re probably better off using Click.\n\n<a name="installation"></a>\n\n## Installation\n\nLethargy only depends on the standard library. You can use [pip](https://pip.pypa.io/en/stable/) to install lethargy.\n\n```bash\npip install lethargy\n```\n\n<a name="usage"></a>\n\n## Usage\n\n```python\nfrom lethargy import Opt\n\n# --use-headers\nheaders = Opt("use headers").take_flag()\n\n# -f|--file <value>\noutput_file = Opt("f", "file").takes(1).take_args()\n```\n\nLethargy returns values appropriate to the option, safely mutating the argument list.\n\n<a name="getting-started"></a>\n\n## Getting Started\n\n<a name="argv"></a>\n\n### The default `argv`\n\nTo save you an additional import, lethargy provides `lethargy.argv` - a clone of the original argument list. Mutating it will not affect `sys.argv`.\n\n**Important note:** `lethargy.argv` is used as a mutable default argument for `lethargy.Opt.take_args` and `lethargy.Opt.take_flag`. Examples below override this value to demonstrate mutation, but in real-world usage, omitting the argument list is recommended (see example in [Usage](#usage)).\n\n<a name="options"></a>\n\n### Options\n\nOptions will automatically convert their names to the appropriate format (`-o` or `--option`). Casing will be preserved.\n\n```python\n>>> from lethargy import Opt\n>>> args = ["-", "--debug", "file.txt"]\n>>> Opt("debug").take_flag(args)\nTrue\n>>> args\n[\'-\', \'file.txt\']\n```\n\nTo set the number of arguments to take, use the `Opt.takes` method.\n\n```python\n>>> args = ["-", "--height", "185cm", "people.csv"]\n>>> Opt("height").takes(1).take_args(args)\n\'185cm\'\n>>> args\n[\'-\', \'people.csv\']\n```\n\nTaking 1 argument will return a single value. Taking multiple will return a list (see the [Argument unpacking](#unpacking) section for details).\n\nYou can also use a "greedy" value, to take every remaining argument. The canonical way to do this is using the Ellipsis literal (`...`).\n\n```python\n>>> args = ["--exclude", ".zshrc", ".bashrc"]\n>>> Opt("exclude").takes(...).take_args(args)\n[\'.zshrc\', \'.bashrc\']\n```\n\n<a name="unpacking"></a>\n\n### Argument unpacking\n\n`lethargy.Opt` makes sure it\'s safe to unpack a returned list of values, unless you override the `default` parameter.\n\n```python\n>>> Opt("x").takes(2).take_args(["-x", "1", "2"])\n[\'1\', \'2\']\n>>> Opt("y").takes(2).take_args([])\n[None, None]\n```\n\nIf there are fewer arguments than expected, `lethargy.ArgsError` will be raised and no mutation will occur. Lethargy has clear and readable error messages.\n\n```python\n>>> args = ["-z", "bad"]\n>>> Opt("z").takes(2).take_args(args)\nTraceback (most recent call last):\n...\nlethargy.ArgsError: expected 2 arguments for \'-z <value> <value>\', found 1 (\'bad\')\n>>> args\n[\'-z\', \'bad\']\n```\n\n<a name="debug-and-verbose"></a>\n\n### `--debug` and `-v`/`--verbose` flags\n\nAs these are such common options, lethargy includes functions out of the box to take these options.\n\n```python\n>>> import lethargy\n>>> args = ["-", "--debug", "--verbose", "sheet.csv"]\n>>> lethargy.take_verbose(args)  # -v or --verbose\nTrue\n>>> lethargy.take_debug(args)\nTrue\n>>> args\n["-", "sheet.csv"]\n```\n\nBy convention, passing `--verbose` will cause a program to output more information. To make implementing this behaviour easier, lethargy has the `print_if` function, which will return `print` if its input is true and a dummy function if not.\n\n```python\nfrom lethargy import take_verbose, print_if\n\nverbose_print = print_if(take_verbose())\n\nverbose_print("This will only print if `--verbose` or `-v` were used!")\n```\n\n<a name="str-and-repr"></a>\n\n### Using `str` and `repr`\n\n`Opt` instances provide a logical and consistent string form.\n\n```python\n>>> str(Opt("flag"))\n\'--flag\'\n>>> str(Opt("e", "example").takes(1))\n\'-e|--example <value>\'\n>>> str(Opt("xyz").takes(...))\n\'--xyz [value]...\'\n```\n\nThe `repr` form makes debugging easy. Note that the order of the names is not guaranteed.\n\n```python\n>>> Opt("f", "flag")\n<Opt(\'-f\', \'--flag\') at 0x106d73f70>\n>>> Opt("example").takes(2)\n<Opt(\'--example\').takes(2) at 0x106ce35e0>\n>>> Opt("test").takes(1, int)\n<Opt(\'--test\').takes(1, int) at 0x106d73f70>\n>>> Opt("x").takes(..., lambda s: s.split())\n<Opt(\'-x\').takes(Ellipsis, <function <lambda> at 0x106ddd9d0>) at 0x106ec0a30>\n```\n\n<a name="raising"></a>\n\n### Raising instead of defaulting\n\nIf `Opt.take_args` is called with `raises=True`, `lethargy.MissingOption` will be raised instead of returning a default, even if the default is set explicitly.\n\nThis behaviour makes it easy to implement mandatory options.\n\n```python\nfrom lethargy import Opt, MissingOption\n\nopt = Opt(\'example\').takes(1)\n\ntry:\n    value = opt.take_args(raises=True)\nexcept MissingOption:\n    print(f\'Missing required option: {opt}\')\n    exit(1)\n```\n\n<a name="conversion"></a>\n\n### Value conversion\n\n`Opt.takes` can optionally take a callable, which will be used to convert the result of `Opt.take_args`. No additional error handling is performed, and the default value will not be converted.\n\n```python\n>>> Opt(\'n\').takes(1, int).take_args([\'-n\', \'28980\'])\n28980\n>>> Opt(\'f\').takes(2, float).take_args([\'-f\', \'1\', \'3.1415\'])\n[1.0, 3.1415]\n>>> Opt(\'chars\').takes(1, set).take_args([])\nNone\n>>> Opt(\'chars\').takes(1, set).take_args([], d=\'Default\')\n\'Default\'\n```\n\n<a name="mutation"></a>\n\n### Disabling mutation\n\n`Opt.take_args` and `Opt.take_flag` both take the optional keyword argument `mut`. Setting `mut` to False disables mutation.\n\n```python\n>>> lst = ["--name", "test",  "example"]\n>>> Opt("name").takes(2).take_args(lst, mut=False)\n[\'test\', \'example\']\n>>> lst  # It hasn\'t changed!\n[\'--name\', \'test\', \'example\']\n```\n\n<a name="contributing"></a>\n\n## Contributing\n\nAny contributions and feedback are welcome! I\'d appreciate it if you could open an issue to discuss changes before submitting a PR, but it\'s not enforced.\n\n<a name="license"></a>\n\n## License\n\nLethargy is released under the [MIT license](https://github.com/SeparateRecords/lethargy/blob/master/LICENSE).\n',
    'author': 'SeparateRecords',
    'author_email': 'me@rob.ac',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/SeparateRecords/lethargy',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
