from __future__ import annotations

from importlib.util import find_spec

from django.db import models

if find_spec("simple_history"):
    from simple_history.models import HistoricalRecords
else:
    HistoricalRecords = None


class TimeStamped(models.Model):
    """
    Abstract model for adding `created_at` and `updated_at` fields.

    This model is based on `TimeStampedModel` from the `django-model-utils`
    package, which can be found at https://github.com/jazzband/django-model-utils.
    License for the original code is below.

    Copyright (c) 2009-2019, Carl Meyer and contributors
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are
    met:

        * Redistributions of source code must retain the above copyright
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above
          copyright notice, this list of conditions and the following
          disclaimer in the documentation and/or other materials provided
          with the distribution.
        * Neither the name of the author nor the names of other
          contributors may be used to endorse or promote products derived
          from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
    A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
    OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
    SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
    DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
    OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
    """

    created_at = models.DateTimeField(auto_now_add=True, editable=False)
    updated_at = models.DateTimeField(auto_now=True, editable=False)

    class Meta:
        abstract = True

    def save(self, *args, **kwargs) -> None:
        """
        Overriding the save method in order to make sure that
        modified field is updated even if it is not given as
        a parameter to the update field argument.
        """
        update_fields = kwargs.get("update_fields", None)
        if update_fields:
            kwargs["update_fields"] = set(update_fields).union({"updated_at"})

        super().save(*args, **kwargs)

    def is_edited(self) -> bool:
        return self.created_at != self.updated_at


class WithHistory(models.Model):
    """
    Abstract model for adding historical records to a model.
    """

    history = HistoricalRecords(inherit=True)

    class Meta:
        abstract = True

    def save(self, *args, without_history: bool = False, **kwargs) -> None:
        if without_history:
            self.save_without_history(*args, **kwargs)
        else:
            super().save(*args, **kwargs)

    def save_without_history(self, *args, **kwargs) -> None:
        self.skip_history_when_saving = True
        try:
            self.save(*args, **kwargs)
        finally:
            del self.skip_history_when_saving
