import tensorflow as tf
from tf_siren.meta import meta_siren


class MetaSIRENModel(tf.keras.Model):

    def __init__(self, input_units: int, hidden_units: int, final_units: int, hyper_units: int,
                 final_activation: str = "linear",
                 hyper_activation: str = "relu",
                 num_layers: int = 1,
                 num_hyper_layers: int = 1,
                 w0: float = 30.0,
                 w0_initial: float = 30.0,
                 kernel_initializer: str = 'he_normal',
                 use_bias: bool = True,
                 kernel_regularizer: float = None,
                 bias_regularizer: float = None,
                 name="MetaSIRENModel",
                 **kwargs):
        """
        Meta SIREN model from the paper [Implicit Neural Representations with Periodic Activation Functions](https://arxiv.org/abs/2006.09661).
        Used to create a meta multi-layer MLP using MetaSinusodialRepresentationDense layers.

        MetaSinusodialRepresentationDense comprises of two components -
        1) The feed forward network that computes the weights for the layer.
        2) The meta layer (inner layer) that will use the above computed weights,
            and perform the actual forward pass of the meta layer.

        Args:
            input_units: Positive integer, dimensionality of the input space.
            hidden_units: Positive integer, dimensionality of the latent space of the
                layers.
            final_units: Positive integer, dimensionality of the output space.
            hyper_units: Positive integer, dimensionality of the hidden space of the
                hyper feed forward network.
            final_activation: Activation function for the meta SinusodialRepresentationDense
                final layer.
            hyper_activation: Activation function for the meta SinusodialRepresentationDense
                layer.
            num_layers: Number of layers in the meta SIREN network.
            num_hyper_layers: Number of layers in the hyper network, per layer of the meta
                SIREN network.
            w0: w0 in the activation step `act(x; w0) = sin(w0 * x)`.
            w0_initial: By default, scales `w0` of first layer to 30 (as used in the paper).
            kernel_initializer: String defining the initializer used for the kernel matrix.
            use_bias: Boolean whether to use bias or not.
            kernel_regularizer: Float defining the regularization strength used for the kernel matrix
                of the hyper network.
            bias_regularizer: Float defining the regularization strength used for the kernel matrix
                of the hyper network.

        # References:
            -   [Implicit Neural Representations with Periodic Activation Functions](https://arxiv.org/abs/2006.09661)
        """
        super().__init__(**kwargs)

        kwargs['name'] = name

        self.input_units = input_units
        self.hidden_units = hidden_units
        self.output_units = final_units
        self.hyper_units = hyper_units
        self.use_bias = use_bias

        hyper_layers = [
            meta_siren.MetaSinusodialRepresentationDense(
                input_units=input_units, output_units=hidden_units, hyper_units=hyper_units,
                num_hyper_layers=num_hyper_layers, hyper_activation=hyper_activation,
                siren_activation='sine', w0=w0_initial, use_bias=use_bias,
                kernel_initializer=kernel_initializer,
                kernel_regularizer=kernel_regularizer,
                bias_regularizer=bias_regularizer, **kwargs)
        ]

        for _ in range(num_layers):
            hyper_layers.append(meta_siren.MetaSinusodialRepresentationDense(
                input_units=hidden_units, output_units=hidden_units, hyper_units=hyper_units,
                num_hyper_layers=num_hyper_layers, hyper_activation=hyper_activation,
                kernel_initializer=kernel_initializer,
                siren_activation='sine', w0=w0, use_bias=use_bias, kernel_regularizer=kernel_regularizer,
                bias_regularizer=bias_regularizer, **kwargs))

        self.hyper_layers = hyper_layers

        self.hyper_final = meta_siren.MetaSinusodialRepresentationDense(
                input_units=hidden_units, output_units=final_units, hyper_units=hyper_units,
                num_hyper_layers=num_hyper_layers, hyper_activation=hyper_activation,
                siren_activation=final_activation, w0=w0, use_bias=use_bias,
                kernel_initializer=kernel_initializer,
                kernel_regularizer=kernel_regularizer,
                bias_regularizer=bias_regularizer, **kwargs)

    @tf.function
    def call(self, context, **kwargs):
        # preserve the parameters generated by the model to pass as output
        layer_params = []
        for layer in self.hyper_layers:
            # construct weight matrix and set to the inner models
            params = layer(context=context)
            layer_params.append(params)

        params = self.hyper_final(context=context)
        layer_params.append(params)
        return layer_params

    @tf.function
    def inner_call(self, inputs, param_list):
        """
        A convenience method to perform a forward pass over the meta layer.
        Requires the weights generated from the call() method to be passed as `params`.

        Args:
            inputs: Input tensors to the meta layer.
            param_list: Parameters of this meta layer.
        """
        x = inputs
        for hyper_layer, params in zip(self.hyper_layers, param_list):
            x = hyper_layer.inner_call(x, params)

        out = self.hyper_final.inner_call(x, param_list[-1])
        return out
