"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const cloud9 = require("../lib");
let stack;
let vpc;
beforeEach(() => {
    stack = new cdk.Stack();
    vpc = new ec2.Vpc(stack, 'VPC');
});
test('create resource correctly with only vpc provide', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', { vpc });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('create resource correctly with both vpc and subnetSelectio', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', {
        vpc,
        subnetSelection: {
            subnetType: ec2.SubnetType.PRIVATE,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('import correctly from existing environment', () => {
    // WHEN
    const c9env = cloud9.Ec2Environment.fromEc2EnvironmentName(stack, 'ImportedEnv', 'existingEnvName');
    // THEN
    expect(c9env).toHaveProperty('ec2EnvironmentName');
});
test('create correctly with instanceType specified', () => {
    // WHEN
    new cloud9.Ec2Environment(stack, 'C9Env', {
        vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.LARGE),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Cloud9::EnvironmentEC2'));
});
test('throw error when subnetSelection not specified and the provided VPC has no public subnets', () => {
    // WHEN
    const privateOnlyVpc = new ec2.Vpc(stack, 'PrivateOnlyVpc', {
        maxAzs: 2,
        subnetConfiguration: [
            {
                subnetType: ec2.SubnetType.ISOLATED,
                name: 'IsolatedSubnet',
                cidrMask: 24,
            },
        ],
    });
    // THEN
    expect(() => {
        new cloud9.Ec2Environment(stack, 'C9Env', {
            vpc: privateOnlyVpc,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.LARGE),
        });
    }).toThrow(/no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection/);
});
//# sourceMappingURL=data:application/json;base64,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