from abc import ABCMeta, abstractmethod
from collections import defaultdict
from dataclasses import dataclass
from typing import Any, Callable, Dict, Generic, List, Optional, Type, TypeVar

from .context import Context
from .finding import Finding, Severity


@dataclass
class ScanMetadata:
    # What service does this scan belong to?
    service: str
    # What is the name of this scan?
    name: str
    # What is the title of the finding generated by this scan?
    title: str
    # What is the severity of the finding?
    severity: Severity

    # A list of IAM permissions required to perform this scan.
    # - use "roles/iam.securityReviewer" by default
    # - add other permissions as needed
    # - if "roles/iam.securityReviewer" is too broad, use the specific permission(s) instead
    permissions: List[str]


class Scan(metaclass=ABCMeta):
    """
    The base class representing a scan to be performed.
    """

    # wrapper around Finding() ctor; kwargs are passed on to markdown template
    def finding(self, **kwargs) -> Finding:
        meta = self.meta()
        return Finding(f"{meta.service}_{meta.name}.md", meta.title, meta.severity, **kwargs)

    @staticmethod
    @abstractmethod
    def meta() -> ScanMetadata:
        raise NotImplementedError("Scans must implement this method")

    @abstractmethod
    def run(self, context: Context) -> Optional[Finding]:
        raise NotImplementedError("Scans must implement this method")


ScanType = Type[Scan]

all_scans: List[ScanType] = []
scans_by_service: Dict[str, List[ScanType]] = defaultdict(lambda: list())
scans_by_name: Dict[str, ScanType] = {}


def scan(s: ScanType) -> ScanType:
    meta = s.meta()

    # Add this scan to the list of all the scans we're able to do.

    all_scans.append(s)
    scans_by_service[meta.service].append(s)
    scans_by_name[meta.name] = s

    return s
