"""
  ____       _ ____  _ _         _____ _____ ____  __  __ _ _         _____           _ _    _ _
 / ___|  ___(_) __ )(_) |_ ___  |_   _| ____|  _ \|  \/  (_) |_ ___  |_   _|__   ___ | | | _(_) |_
 \___ \ / __| |  _ \| | __/ _ \   | | |  _| | |_) | |\/| | | __/ _ \   | |/ _ \ / _ \| | |/ / | __|
  ___) | (__| | |_) | | ||  __/   | | | |___|  _ <| |  | | | ||  __/   | | (_) | (_) | |   <| | |_
 |____/ \___|_|____/|_|\__\___|   |_| |_____|_| \_\_|  |_|_|\__\___|   |_|\___/ \___/|_|_|\_\_|\__|

Preprocessing functions- using your TERMite output to make AI-ready data

"""

__author__ = 'SciBite DataScience'
__version__ = '0.2'
__copyright__ = '(c) 2019, SciBite Ltd'
__license__ = 'Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License'

import termite_toolkit.termite as termite
import json


def get_hits(termiteTags, hierarchy=None, vocabs=None):
    '''
    Helper function. Uses termiteTags and hierarchy to collect info on the highest priority hits.

    :param array termiteTags: Locations of TERMite hits found, extracted from the TERMite json
    :param dict hierarchy: Dictionary with a hierarchy of vocabs to prioritise in case of overlap
    :param array(str) vocabs: List of vocabs to be substituted, ordered by priority. These vocabs MUST be in the TERMite results. If left
    empty, all vocabs found will be used with random priority where overlaps are found.
    :return array(dict):
    '''
    hits = []
    for hit in termiteTags:
        if not vocabs:
            if hit['entityType'] not in hierarchy:
                hierarchy[hit['entityType']] = len(hierarchy)
        else:
            if hit['entityType'] not in vocabs:
                continue

        hitLocs, subsumeStates = hit['exact_array'], hit['subsume']
        assert len(hitLocs) == len(subsumeStates)

        for idx, hitLoc in enumerate(hitLocs):
            if hitLoc['fls'][0] < 1:
                continue
            hitInfo = {}
            hitInfo['entityType'], hitInfo['entityID'], hitInfo['entityName'] = hit['entityType'], hit['hitID'], hit[
                'name']
            breakBool = False
            hitInfo['startLoc'], hitInfo['endLoc'] = hitLoc['fls'][1], hitLoc['fls'][2]
            if subsumeStates[idx] == False:  # If hit is not subsumed...
                for hitIdx, hit_ in enumerate(hits):
                    # Compare to already found hits to check there's no conflict
                    if ((hit_['endLoc'] >= hitInfo['startLoc'] and hit_['endLoc'] <= hitInfo['endLoc']) or
                            (hit_['startLoc'] >= hitInfo['startLoc'] and hit_['startLoc'] <= hitInfo['endLoc'])):
                        # If they overlap, check their position in the hierarchy
                        if hierarchy[hit_['entityType']] >= hierarchy[hitInfo['entityType']]:
                            del hits[hitIdx]
                            break
                        else:
                            breakBool = True
                            break
            if not breakBool:
                hits.append(hitInfo)
    return hits


def markup(docjsonx, normalisation='id', substitute=True, wrap=False,
           wrapChars=('{!', '!}'), vocabs=None, labels=None, replacementDict=None):
    '''
    Receives TERMite docjsonx output. Processes the original text, normalising identified hits.

    :param str docjsonx: JSON string generated by TERMite. Must be docjsonx.
    :param str normalisation: Type of normalisation to substitute/add (must be 'id', 'type', 'name', 'typeplusname' or 'typeplusid')
    :param bool substitute: Whether to replace the found term (or add normalisation alongside)
    :param bool wrap: Whether to wrap found hits with 'bookends'
    :param tuple(str) wrapChars: Tuple of length 2, containing strings to insert at start/end of found hits
    :param array(str) vocabs: List of vocabs to be substituted, ordered by priority. These vocabs MUST be in the TERMite results. If left
    empty, all vocabs found will be used with random priority where overlaps are found.
    :param dict replacementDict: Dictionary with <VOCAB>:<string_to_replace_hits_in_vocab>. '~ID~' will be replaced with the entity id,
    and '~TYPE~' will be replaced with the vocab name. Example: {'GENE':'ENTITY_~TYPE~_~ID~'} would result in BRCA1 -> ENTITY_GENE_BRCA1.
    replacementDict supercedes normalisation. ~NAME~ can also be used to get the preferred name.
    :return dict:
    '''

    results = {}

    validTypes = ['id', 'type', 'name', 'typeplusname', 'typeplusid']
    if normalisation not in validTypes:
        raise ValueError(
            'Invalid normalisation requested. Valid options are \'id\', \'name\', \'type\', \'typeplusname\' and \'tyeplusid\'.')
    if len(wrapChars) != 2 or not all(isinstance(wrapping, str) for wrapping in wrapChars):
        raise ValueError('wrapChars must be a tuple of length 2, containing strings.')
    if labels:
        if labels not in ['word', 'char']:
            raise ValueError('labels, if specified, must be either \'word\' or \'char\'')

    hierarchy = {}
    if vocabs:
        for idx, vocab in enumerate(vocabs):
            hierarchy[vocab] = idx

    if isinstance(docjsonx, str):
        j = json.loads(docjsonx)
    else:
        j = docjsonx

    for docIdx, doc in enumerate(j):
        text = doc['body']

        try:
            substitutions = get_hits(doc['termiteTags'], hierarchy=hierarchy, vocabs=vocabs)
        except KeyError:
            results[docIdx] = {'termited_text': text}
            continue

        if len(substitutions) > 0:
            substitutions.sort(key=lambda x: x['startLoc'])
            substitutions = reversed(substitutions)

        if wrap:
            prefix = wrapChars[0]
            postfix = wrapChars[1]
        else:
            prefix, postfix = '', ''

        for sub in substitutions:
            subText = ''
            if replacementDict:
                subText = replacementDict[sub['entityType']].replace(
                    '~TYPE~', sub['entityType']).replace('~ID~', sub['entityID']).replace('~NAME~', sub['entityName'])
            elif normalisation == 'id':
                subText = '_'.join([sub['entityType'], sub['entityID']])
                if not substitute:
                    subText += ' %s' % text[sub['startLoc']:sub['endLoc']]
            elif normalisation == 'type':
                subText = sub['entityType']
                if not substitute:
                    subText += ' %s' % text[sub['startLoc']:sub['endLoc']]
            elif normalisation == 'name':
                subText = sub['entityName']
                if not substitute:
                    subText += ' %s' % text[sub['startLoc']:sub['endLoc']]
            elif normalisation == 'typeplusname':
                subText = '%s %s' % (sub['entityType'], sub['entityName'])
                if not substitute:
                    subText += ' %s' % text[sub['startLoc']:sub['endLoc']]
            elif normalisation == 'typeplusid':
                subText = '%s %s' % (sub['entityType'], '_'.join([sub['entityType'], sub['entityID']]))
                if not substitute:
                    subText += ' %s' % text[sub['startLoc']:sub['endLoc']]
            text = text[:sub['startLoc']] + prefix + subText + postfix + text[sub['endLoc']:]

        results[docIdx] = {'termited_text': text}

    return results


def text_markup(text, termiteAddr='http://localhost:9090/termite', vocabs=['GENE', 'INDICATION', 'DRUG'],
                normalisation='id', wrap=False, wrapChars=('{!', '!}'), substitute=True, replacementDict=None):
    '''
    Receives plain text, returns text with TERMited substitutions.

    :param str normalisation: Type of normalisation to substitute/add (must be 'id', 'type', 'name', 'typeplusname' or 'typeplusid')
    :param bool substitute: Whether to replace the found term (or add normalisation alongside)
    :param bool wrap: Whether to wrap found hits with 'bookends'
    :param tuple(str) wrapChars: Tuple of length 2, containing strings to insert at start/end of found hits
    :param array(str) vocabs: List of vocabs to be substituted, ordered by priority. These vocabs MUST be in the TERMite results. If left
    empty, all vocabs found will be used with random priority where overlaps are found.
    :param dict replacementDict: Dictionary with <VOCAB>:<string_to_replace_hits_in_vocab>. '~ID~' will be replaced with the entity id,
    and '~TYPE~' will be replaced with the vocab name. Example: {'GENE':'ENTITY_~TYPE~_~ID~'} would result in BRCA1 -> ENTITY_GENE_BRCA1
    replacementDict supercedes normalisation. ~NAME~ can also be used to get the preferred name.
    :return str:
    '''

    t = termite.TermiteRequestBuilder()
    t.set_url(termiteAddr)
    t.set_text(text)
    t.set_entities(','.join(vocabs))
    t.set_subsume(True)
    t.set_input_format("txt")
    t.set_output_format("doc.jsonx")
    docjsonx = t.execute()

    return markup(docjsonx, vocabs=vocabs, normalisation=normalisation, wrap=wrap,
                  wrapChars=wrapChars, substitute=substitute, replacementDict=replacementDict)[0]['termited_text']


def label(docjsonx, vocabs, labelLevel='word'):
    '''
    Receives TERMite output docjsonx and returns split text with labels as to what entities are found in that part of the text.

    :param str docjsonx: JSON string generated by TERMite. Must be docjsonx.
    :param str labelLevel: Labels for where hits are found in the text. Must be 'char' or 'word', word by default
    :param array(str) vocabs: List of vocabs to be substituted, ordered by priority. These vocabs MUST be in the TERMite results. If left
    empty, all vocabs found will be used with random priority where overlaps are found.
    :return dict:
    '''

    results = {}
    substitutions = []
    hierarchy = {}
    for idx, vocab in enumerate(vocabs):
        hierarchy[vocab] = idx

    if isinstance(docjsonx, str):
        j = json.loads(docjsonx)
    else:
        j = docjsonx

    for docIdx, doc in enumerate(j):
        text = doc['body']

        splitText, labels = None, None
        if labelLevel == 'word':
            splitText = text.split()
        elif labelLevel == 'char':
            splitText = list(text)
        labels = [0 for i in splitText]

        try:
            hits = get_hits(doc['termiteTags'], hierarchy=hierarchy, vocabs=vocabs)
        except KeyError:
            results[docIdx] = {'split_text': splitText, 'labels': labels}
            continue

        for hit in hits:
            if labelLevel == 'char':
                for i in range(hit['startLoc'], hit['endLoc']):
                    labels[i] = hierarchy[hit['entityType']] + 1
            elif labelLevel == 'word':
                cursor = 0
                for wIdx, w in enumerate(splitText):
                    if cursor >= hit['startLoc'] and cursor <= hit['endLoc']:
                        labels[wIdx] = hierarchy[hit['entityType']] + 1
                    cursor += len(w) + 1

        results[docIdx] = {'split_text': splitText, 'labels': labels}

    return results
