"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
const util_1 = require("./util");
test('ignorefile synthesises correctly', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    // WHEN
    new __1.IgnoreFile(prj, '.dockerignore');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.dockerignore'])).toEqual(['']);
});
test('ignorefile puts includes (!) after excludes and sorts the entries', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    // WHEN
    const file = new __1.IgnoreFile(prj, '.dockerignore');
    file.include('c.txt', 'd.txt');
    file.exclude('a.txt', 'b.txt');
    file.exclude('e.txt', 'f.txt');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.dockerignore'])).toEqual([
        'a.txt',
        'b.txt',
        'e.txt',
        'f.txt',
        '!c.txt',
        '!d.txt',
        '',
    ]);
});
test('ignorefile includes file after exclusion and inclusion', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    // WHEN
    const file = new __1.IgnoreFile(prj, '.dockerignore');
    file.exclude('a.txt');
    file.include('a.txt');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.dockerignore'])).toEqual([
        '!a.txt',
        '',
    ]);
});
test('ignorefile excludes file after inclusion and exclusion', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    // WHEN
    const file = new __1.IgnoreFile(prj, '.dockerignore');
    file.include('a.txt');
    file.exclude('a.txt');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.dockerignore'])).toEqual([
        '!a.txt',
        '',
    ]);
});
test('ignorefile omits duplicated includes and excludes', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    // WHEN
    const file = new __1.IgnoreFile(prj, '.dockerignore');
    file.exclude('a.txt', 'b.txt');
    file.include('c.txt', 'd.txt');
    file.exclude('a.txt', 'b.txt');
    file.include('c.txt', 'd.txt');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.dockerignore'])).toEqual([
        'a.txt',
        'b.txt',
        '!c.txt',
        '!d.txt',
        '',
    ]);
});
test('includes (!) are always at the end', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    const ignore = new __1.IgnoreFile(prj, '.myignorefile');
    // WHEN
    ignore.addPatterns('*.foo');
    ignore.addPatterns('!hello.foo');
    ignore.include('negated.txt');
    ignore.addPatterns('*.bar');
    ignore.exclude('*.bar');
    ignore.include('zoo.foo');
    ignore.addPatterns('!negated.txt');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.myignorefile'])).toStrictEqual([
        '*.bar',
        '*.foo',
        '!hello.foo',
        '!negated.txt',
        '!zoo.foo',
        '',
    ]);
});
test('if include() is called with "!", then strip it', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    const ignore = new __1.IgnoreFile(prj, '.myignorefile');
    // WHEN
    ignore.include('!*.js');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.myignorefile'])).toStrictEqual([
        '!*.js',
        '',
    ]);
});
test('removePatters() can be used to remove previously added patters', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    const ignore = new __1.IgnoreFile(prj, '.myignorefile');
    // WHEN
    ignore.addPatterns('*.js');
    ignore.addPatterns('my_file');
    ignore.addPatterns('!boom/bam');
    ignore.removePatterns('*.zz', '*.js', '!boom/bam');
    ignore.addPatterns('*.zz');
    ignore.addPatterns('boom/bam');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.myignorefile'])).toStrictEqual([
        '*.zz',
        'boom/bam',
        'my_file',
        '',
    ]);
});
test('comments are filtered out', () => {
    // GIVEN
    const prj = new util_1.TestProject();
    const ignore = new __1.IgnoreFile(prj, '.myignorefile');
    // WHEN
    ignore.addPatterns('*.js', '#comment');
    ignore.addPatterns('!foo');
    ignore.addPatterns('# hello world');
    ignore.addPatterns('bar');
    // THEN
    expect(splitAndIgnoreMarker(util_1.synthSnapshot(prj)['.myignorefile'])).toStrictEqual([
        '*.js',
        'bar',
        '!foo',
        '',
    ]);
});
// parses file contents without 'Generated by...' spiel
function splitAndIgnoreMarker(fileContents) {
    return fileContents.split('\n').slice(1);
}
//# sourceMappingURL=data:application/json;base64,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