import abc
import os
import sys

import glob
import gzip
import multiprocessing
import re
import uuid

from functools import cached_property

from searchkit.log import log


class FileSearchException(Exception):
    def __init__(self, msg):
        self.msg = msg


class StateChangeException(FileSearchException):
    """ Used when state is changed unexpectedly in a ThreadProtectedObject """


class ThreadProtectedObject(abc.ABC):
    """
    Denotes an object that is not thread safe and cannot have its state changed
    by a forked process unless that state declares itself as thread safe.

    State changes to this object that are not made at the parent process level
    will be lost as soon as the forked process terminates.
    """

    def __init__(self):
        self._pid = os.getpid()

    @abc.abstractproperty
    def thread_safe_attributes(self):
        """ Returns a list of attributes that are considered thread safe in
        the sense that their state can be changed by a forked process and
        not impact others or be required by the main process.

        Convention used here is that a matched attribute can prefixed with '_'
        or not but the name used here is without prefix '_'. This allows e.g.
        a property and corresponding attribute to be registered in one go.
        """

    def __setattr__(self, key, val):
        if hasattr(self, '_pid'):
            keys = []
            for k in self.thread_safe_attributes:
                keys.extend([k, '_' + k])

            if key not in keys:
                if os.getpid() != self._pid:
                    msg = ("{} state must not be changed by child process "
                           "({}, {})".
                           format(self.__class__.__name__, key, val))
                    raise StateChangeException(msg)

        return super().__setattr__(key, val)


class SearchDefBase(ThreadProtectedObject):

    def __init__(self, constraints=None):
        # preload
        self.id
        self.constraints = {c.id: c for c in constraints or {}}
        # do this last
        super().__init__()

    @cached_property
    def id(self):
        """
        A unique identifier for this search definition.
        """
        return str(uuid.uuid4())

    @property
    def thread_safe_attributes(self):
        return []

    def apply_constraints(self, line):
        """
        Apply any constraints for this searchdef.
        """
        if not self.constraints:
            return True

        for c in self.constraints.values():
            if c.apply_to_line(line):
                continue

            return False

        return True


class SearchDef(SearchDefBase):

    def __init__(self, pattern, tag=None, hint=None,
                 store_result_contents=True, **kwargs):
        """
        Add a search definition.

        @param pattern: regex pattern or list of patterns to search for
        @param tag: optional user-friendly identifier for this search term
        @param hint: pre-search term to speed things up
        @param store_result_contents: by default the content of a search result
                                      is saved but if it is not needed this
                                      can be set to False. This effectively
                                      makes the result True/False.
        """
        if type(pattern) != list:
            self.patterns = [re.compile(pattern)]
        else:
            self.patterns = []
            for _pattern in pattern:
                self.patterns.append(re.compile(_pattern))

        self.store_result_contents = store_result_contents
        self.tag = tag
        if hint:
            self.hint = re.compile(hint)
        else:
            self.hint = None

        # do this last
        super().__init__(**kwargs)

    @property
    def thread_safe_attributes(self):
        return super().thread_safe_attributes

    def run(self, line):
        """Execute search patterns against line and return first match."""
        if self.hint:
            ret = self.hint.search(line)
            if not ret:
                return None

        ret = None
        for pattern in self.patterns:
            ret = pattern.match(line)
            if ret:
                return ret

        return ret


class SequenceSearchDef(SearchDefBase):

    def __init__(self, start, tag, end=None, body=None, **kwargs):
        """
        Define search for sequences. A sequence must match a start and end with
        optional body in between. If no end defined, the sequence ends with
        the start of the next or EOF.

        NOTE: sequences must not overlap. This is therefore not suitable for
        finding sequences generated by parallel/concurrent tasks.

        @param start: SearchDef object for matching start
        @param tag: tag used to identify this sequence definition
        @param end: optional SearchDef object for matching end
        @param body: optional SearchDef object for matching body
        """
        self.s_start = start
        self.s_end = end
        self.s_body = body
        self.tag = tag
        self._mark = None
        # Each section identified gets its own id. Since each file is processed
        # using a separate process and memory is not shared, these values must
        # be unique to avoid collisions when results are aggregated.
        self._section_id = None
        # do this last
        super().__init__(**kwargs)

    @property
    def thread_safe_attributes(self):
        # these are safe to change within a process context since their
        # state doesn't count towards the final result and is not expected to
        # be shared.
        attrs = super().thread_safe_attributes
        return attrs + ['section_id', 'mark']

    @property
    def start_tag(self):
        """Tag used to identify start of section"""
        return "{}-start".format(self.tag)

    @property
    def end_tag(self):
        """Tag used to identify end of section"""
        return "{}-end".format(self.tag)

    @property
    def body_tag(self):
        """Tag used to identify body of section"""
        return "{}-body".format(self.tag)

    @property
    def section_id(self):
        """ ID of current section. A new id should be set after each
        completed section. """
        return self._section_id

    @property
    def started(self):
        """Indicate a section sequence has been started."""
        return self._mark == 1

    def start(self):
        """Indicate that a sequence start has been detected."""
        self._section_id = str(uuid.uuid4())
        self._mark = 1

    def reset(self):
        """Used to restart a section. This is used e.g. if the start
        expression matches midway through a sequence (and before the end).
        """
        self._mark = 0

    def stop(self):
        """Indicate that a sequence is complete."""
        self._mark = 0
        self._section_id = str(uuid.uuid4())


class SearchResultPart(object):

    def __init__(self, index, value):
        self.index = index
        self.value = value


class SearchResult(object):

    def __init__(self, linenumber, source, result, search_term_tag=None,
                 section_id=None, sequence_obj_id=None, store_contents=True):
        """
        @param linenumber: line number that produced a match
        @param source: data source (path)
        @param result: python.re match object
        @param search_term_tag: SearchDef object tag
        @param section_id: SequenceSearchDef object section id
        @param sequence_obj_id: SequenceSearchDef object unique id
        """
        self.tag = search_term_tag
        self.source = source
        self.linenumber = linenumber
        self._parts = {}
        self.sequence_obj_id = sequence_obj_id
        self.section_id = section_id

        if not store_contents:
            log.debug("store_contents is False - skipping save")
            return

        num_groups = len(result.groups())
        # NOTE: this does not include group(0)
        if num_groups:
            # To reduce memory footprint, don't store group(0) i.e. the whole
            # line, if there are actual groups in the result.
            for i in range(1, num_groups + 1):
                self._add(i, result.group(i))
        else:
            log.debug("saving full search result which can lead to high "
                      "memory usage")
            self._add(0, result.group(0))

    def _add(self, index, value):
        self._parts[index] = SearchResultPart(index, value)

    def get(self, index):
        """Retrieve a result part by its index."""
        if index not in self._parts:
            return None

        return self._parts[index].value

    def __len__(self):
        return len(self._parts)

    def __iter__(self):
        for idx in sorted(self._parts.keys()):
            yield self._parts[idx].value

    def __repr__(self):
        r_list = ["{}={}".format(k, v.value) for k, v in self._parts.items()]
        return "ln:{} ".format(self.linenumber) + ", ".join(r_list)


class SearchResultsCollection(object):

    def __init__(self):
        self.reset()

    def __len__(self):
        _count = 0
        for f in self.files:
            _count += len(self.find_by_path(f))

        return _count

    @property
    def files(self):
        return list(self._results.keys())

    def reset(self):
        self._iter_idx = 0
        self._results = {}

    def add(self, path, results):
        if path not in self._results:
            self._results[path] = results
        else:
            self._results[path] += results

    def find_by_path(self, path):
        if path not in self._results:
            return []

        return self._results[path]

    def find_by_tag(self, tag, path=None, sequence_obj_id=None):
        """Return all result tagged with tag.

        If no path is provided tagged results from all paths are returned.
        """
        if path:
            paths = [path]
        else:
            paths = list(self._results.keys())

        results = []
        for path in paths:
            for result in self._results.get(path, []):
                if sequence_obj_id is None:
                    if result.tag == tag:
                        results.append(result)
                else:
                    if (result.tag == tag and
                            result.sequence_obj_id == sequence_obj_id):
                        results.append(result)

        return results

    def find_sequence_sections(self, sequence_obj, path=None):
        """Return results of running the given sequence search.

        Returns a dictionary keyed by section id where each is a list of
        results for that section with start, body, end etc.
        """
        _results = []
        sections = {}
        _results += self.find_by_tag(tag=sequence_obj.start_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        _results += self.find_by_tag(tag=sequence_obj.body_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        _results += self.find_by_tag(tag=sequence_obj.end_tag, path=path,
                                     sequence_obj_id=sequence_obj.id)
        for r in _results:
            if r.section_id in sections:
                sections[r.section_id].append(r)
            else:
                sections[r.section_id] = [r]

        return sections

    def __iter__(self):
        return iter(self._results.items())


class SearcherBase(abc.ABC):

    @abc.abstractmethod
    def add(self, searchdef):
        """
        Add a search criterea.

        @param searchdef: SearchDef object
        """

    @abc.abstractproperty
    def files(self):
        """ Returns a list of files we will be searching. """

    @abc.abstractproperty
    def num_parallel_tasks(self):
        """
        Returns an integer representing the maximum number of tasks we can
        run in parallel. This will typically be bound by the number of
        cpu threads available.
        """

    @abc.abstractmethod
    def run(self):
        """
        Execute all searches.
        """


class FileSearcher(SearcherBase):

    def __init__(self, max_parallel_tasks=8, max_logrotate_depth=7,
                 constraint=None):
        """
        @param max_parallel_tasks: TODO
        @param max_logrotate_depth: TODO
        @param constraint: global constraint to be used with this
                           searcher that applies to all files searched.
        """
        self.max_parallel_tasks = max_parallel_tasks
        self.max_logrotate_depth = max_logrotate_depth
        self.constraint = constraint
        self.global_constraint_restrictions = {}
        self.paths = {}
        self.results = SearchResultsCollection()
        log.debug("filesearcher: created (global constraint=%s)",
                  self.constraint is not None)

    @property
    def num_parallel_tasks(self):
        if self.max_parallel_tasks == 0:
            cpus = 1  # i.e. no parallelism
        else:
            cpus = min(self.max_parallel_tasks, os.cpu_count())

        return min(len(self.files), cpus)

    def add(self, searchdef, path, allow_global_constraints=True):
        """Add a term to search for.

        A search definition is registered against a path which can be a
        file,  directory or glob. Any number of searches can be registered.
        Searches are executed concurrently by file.

        @param searchdef: SearchDef object
        @param path: path that we will be searching for this key
        @param allow_global_constraints: whether to allow global constraints to
                                         be applied to this path.
        """
        log.debug("filesearcher: registered search with tag %s", searchdef.id)
        if path in self.paths:
            self.paths[path].append(searchdef)
        else:
            self.paths[path] = [searchdef]

        if not allow_global_constraints:
            self.global_constraint_restrictions[path] = True

    def _job_wrapper(self, pool, path, entry):
        term_key = path
        return pool.apply_async(self._search_task_wrapper,
                                (entry, term_key))

    def _search_task_wrapper(self, path, term_key):
        if os.path.getsize(path) == 0:
            log.debug("skipping zero-length file %s", path)
            return

        try:
            with gzip.open(path, 'rb') as fd:
                try:
                    # test if file is gzip
                    fd.read(1)
                    fd.seek(0)
                    return self._search_task(term_key, fd, path)
                except OSError:
                    pass

            with open(path, 'rb') as fd:
                return self._search_task(term_key, fd, path)
        except UnicodeDecodeError:
            log.exception("")
            # ignore the file if it can't be decoded
            log.debug("caught UnicodeDecodeError for path %s - skipping", path)
        except EOFError as e:
            log.exception("")
            msg = ("an exception occured while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e
        except Exception as e:
            log.exception("")
            msg = ("an unknown exception occurred while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e

    def apply_global_constraints(self, fd):
        offset = 0
        if not self.constraint:
            log.debug("no global constraint to apply to %s", fd.name)
            return offset

        if fd.name in self.global_constraint_restrictions:
            log.debug("skipping global constraint for %s", fd.name)
            return offset

        # if not os.path.isdir(os.path.join(HotSOSConfig.data_root,
        #                                   'sos_commands')):
        #     log.info("skipping global constraints since data_root is not a "
        #              "sosreport therefore files may be changing")
        #     return offset

        log.debug("applying global constraint %s to %s", self.constraint,
                  fd.name)
        _offset = self.constraint.apply_to_file(fd)
        if _offset is not None:
            return _offset

        return offset

    def _search_task(self, term_key, fd, path):
        results = []
        sequence_results = {}
        offset = self.apply_global_constraints(fd)
        # mark all as runnable to start
        search_defs = {s_term: True for s_term in self.paths[term_key]}
        sd_constraints = []
        for sd in search_defs:
            if sd.constraints:
                for c in sd.constraints.values():
                    sd_constraints.append(c)

                # mark ones with constraints as not runnable until constraint
                # is met.
                search_defs[sd] = False

        log.debug("starting search of %s (offset=%s, pos=%s)", path, offset,
                  fd.tell())
        if sd_constraints:
            constraints_msg = "applying constraints to {}:".format(path)
            for c in sd_constraints:
                constraints_msg += "\n  {}".format(c)

            log.debug(constraints_msg)

        lines_searched = 0
        # NOTE: line numbers start at 1 hence  offset + 1
        for ln, line in enumerate(fd, start=offset + 1):
            lines_searched += 1
            if type(line) == bytes:
                line = line.decode("utf-8")

            for s_term, runnable in search_defs.items():
                if not runnable:
                    if not s_term.apply_constraints(line):
                        continue

                    search_defs[s_term] = True

                if type(s_term) == SequenceSearchDef:
                    # if the ending is defined and we match a start while
                    # already in a section, we start again.
                    if s_term.s_end:
                        ret = s_term.s_start.run(line)
                        if s_term.started:
                            if ret:
                                # reset and start again
                                if sequence_results:
                                    del sequence_results[s_term.id]

                                s_term.reset()
                            else:
                                ret = s_term.s_end.run(line)
                    else:
                        ret = s_term.s_start.run(line)
                else:
                    ret = s_term.run(line)

                if ret:
                    section_id = None
                    sequence_obj_id = None
                    tag = s_term.tag
                    store_contents = True
                    if type(s_term) == SequenceSearchDef:
                        if not s_term.started:
                            tag = s_term.start_tag
                            s_term.start()
                            section_id = s_term.section_id
                        else:
                            tag = s_term.end_tag
                            section_id = s_term.section_id
                            s_term.stop()
                            # if no end is defined then we dont bother storing
                            # the result, just complete the section and start
                            # the next.
                            if s_term.s_end is None:
                                tag = s_term.start_tag
                                s_term.start()
                                section_id = s_term.section_id

                        sequence_obj_id = s_term.id
                    else:
                        store_contents = s_term.store_result_contents

                    r = SearchResult(ln, path, ret, tag, section_id=section_id,
                                     sequence_obj_id=sequence_obj_id,
                                     store_contents=store_contents)
                    if type(s_term) == SequenceSearchDef:
                        if s_term.id not in sequence_results:
                            sequence_results[s_term.id] = [r]
                        else:
                            sequence_results[s_term.id].append(r)
                    else:
                        results.append(r)

                elif type(s_term) == SequenceSearchDef:
                    if s_term.started and s_term.s_body:
                        ret = s_term.s_body.run(line)
                        if not ret:
                            continue

                        r = SearchResult(ln, path, ret, s_term.body_tag,
                                         section_id=s_term.section_id,
                                         sequence_obj_id=s_term.id)
                        sequence_results[s_term.id].append(r)

        if sequence_results:
            # If a sequence ending definition is provided and we reached EOF
            # while a sequence is started, complete the sequence if s_end
            # matches an empty string. If s_end is not defined we just go ahead
            # and complete the section.
            filter_section_id = {}
            for s_term in search_defs:
                if type(s_term) == SequenceSearchDef:
                    if s_term.started:
                        if s_term.s_end is None:
                            s_term.stop()
                        else:
                            ret = s_term.s_end.run("")
                            if ret:
                                section_id = s_term.section_id
                                s_term.stop()
                                tag = s_term.end_tag
                                r = SearchResult(ln + 1, path, ret, tag,
                                                 section_id=section_id,
                                                 sequence_obj_id=s_term.id)
                            else:
                                if s_term.id not in filter_section_id:
                                    filter_section_id[s_term.id] = []

                                filter_section_id[s_term.id].append(
                                    s_term.section_id)

            # Now add sequece results to main results list, excluding any
            # incomplete sections.
            for s_results in sequence_results.values():
                for r in s_results:
                    if filter_section_id:
                        seq_id = r.sequence_obj_id
                        if (seq_id is not None and
                                seq_id in filter_section_id):
                            if r.section_id in filter_section_id[seq_id]:
                                continue

                    results.append(r)

        log.debug("completed search of %s lines", lines_searched)
        if sd_constraints:
            constraints_msg = "constraints stats {}:".format(path)
            for c in sd_constraints:
                constraints_msg += "\n  id={}: {}".format(c.id, c.stats())

            log.debug(constraints_msg)

        return results

    def logrotate_file_sort(self, fname):
        """
        This is used to sort the contents of a directory by passing the
        function as a the key to a list sort.

        Assumes that the filenames use logrotate format i.e. .log, .log.1,
        .log.2.gz etc.
        """

        filters = [r"\S+\.log$",
                   r"\S+\.log\.(\d+)$",
                   r"\S+\.log\.(\d+)\.gz?$"]
        for filter in filters:
            ret = re.compile(filter).match(fname)
            if ret:
                break

        # files that dont follow logrotate naming format go to the end.
        if not ret:
            # put at the end
            return 100000

        if len(ret.groups()) == 0:
            return 0

        return int(ret.group(1))

    def filtered_paths(self, paths):
        """
        Paths can be a mix of files and directories.
        """
        logrotate_collection = {}
        dir_contents = []
        for path in paths:
            if not os.path.isfile(path):
                continue

            ret = re.compile(r"(\S+)\.log\S*").match(path)
            if ret:
                base = ret.group(1)
                if base not in logrotate_collection:
                    logrotate_collection[base] = []

                if re.compile(r"(\S+)\.log\S+").match(path):
                    logrotate_collection[base].append(path)
                else:
                    dir_contents.append(base + '.log')
            else:
                dir_contents.append(path)

        limit = self.max_logrotate_depth
        for logrotated in logrotate_collection.values():
            capped = sorted(logrotated,
                            key=self.logrotate_file_sort)[:limit]
            dir_contents += capped

        return dir_contents

    @property
    def files(self):
        _files = []
        for user_path in self.paths:
            if os.path.isfile(user_path):
                _files.append(user_path)
            elif os.path.isdir(user_path):
                _files.extend(self.filtered_paths(user_path))
            else:
                _files.extend(self.filtered_paths(glob.glob(user_path)))

        return _files

    def run(self):
        """Execute all the search queries.

        @return: search results
        """
        self.results.reset()
        if not self.paths:
            # If no searches have been registered we don't have anything to do.
            log.debug("filesearcher: no search terms registered so nothing to "
                      "do")
            return self.results

        num_files = len(self.files)
        if not num_files:
            log.debug("filesearcher: no files to search")
            return self.results

        with multiprocessing.Pool(processes=self.num_parallel_tasks) as pool:
            jobs = {}
            for user_path in self.paths:
                jobs[user_path] = []
                if os.path.isfile(user_path):
                    job = self._job_wrapper(pool, user_path, user_path)
                    jobs[user_path] = [(user_path, job)]
                elif os.path.isdir(user_path):
                    for path in self.filtered_paths(user_path):
                        if user_path in self.global_constraint_restrictions:
                            self.global_constraint_restrictions[path] = True

                        job = self._job_wrapper(pool, user_path, path)
                        jobs[user_path].append((path, job))
                else:
                    for path in self.filtered_paths(glob.glob(user_path)):
                        if user_path in self.global_constraint_restrictions:
                            self.global_constraint_restrictions[path] = True

                        job = self._job_wrapper(pool, user_path, path)
                        jobs[user_path].append((path, job))

            num_searches = sum([len(jobs[p]) * len(self.paths[p])
                                for p in jobs])
            log.debug("filesearcher: running processes=%d files=%d "
                      "searches=%d", self.num_parallel_tasks,
                      num_files, num_searches)

            for user_path in jobs:
                for fpath, job in jobs[user_path]:
                    try:
                        result = job.get()
                        if result:
                            self.results.add(fpath, result)
                    except FileSearchException as e:
                        sys.stderr.write("{}\n".format(e.msg))

        log.debug("filesearcher: completed (results=%s)", len(self.results))
        return self.results
