"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FsxOntap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
class FsxOntap extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.mountPath = '/mnt/fsx';
        this.mountName = '/datavol';
        const name = props.name ?? 'fsx-ontap';
        if (props.mountName) {
            this.mountName = props.mountName;
        }
        if (props.mountPath) {
            this.mountPath = props.mountPath;
        }
        const vpc = props.vpc;
        /**
        * Amazon FSx for NetApp ONTAP accepts either one or two subnets.
        * NB, this construct requires them to be private ones.
        * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html
        */
        const privateSubnetsForFsx = vpc.privateSubnets.slice(0, 2);
        if (privateSubnetsForFsx.length !== 2 && privateSubnetsForFsx.length !== 1) {
            throw new Error('FSx for NetApp ONTAP accepts either one or two subnets. Moreover, they must be private subnets!');
        }
        // Create a random password for the FSx for NetApp ONTAP admin user.
        const fsxAdminPassword = new aws_secretsmanager_1.Secret(this, `${name}`, {
            generateSecretString: {
                excludePunctuation: true,
            },
        });
        const fsxSecurityGroup = new aws_ec2_1.SecurityGroup(this, `${name}-fsx-security-group`, {
            vpc,
            description: 'The security group for the FSx for NetApp ONTAP service',
            allowAllOutbound: true,
        });
        this.addIngressToFsxFromSourceSg(props.securityGroupSource, fsxSecurityGroup);
        const cfnFileSystem = new aws_cdk_lib_1.aws_fsx.CfnFileSystem(this, `${name}-filesystem`, {
            fileSystemType: 'ONTAP',
            subnetIds: privateSubnetsForFsx.map(s => s.subnetId),
            ontapConfiguration: {
                fsxAdminPassword: fsxAdminPassword.secretValue.toString(),
                deploymentType: privateSubnetsForFsx.length === 2 ? 'MULTI_AZ_1' : 'SINGLE_AZ_1',
                diskIopsConfiguration: {
                    iops: 40000,
                    mode: 'USER_PROVISIONED',
                },
                preferredSubnetId: privateSubnetsForFsx[0].subnetId,
                routeTableIds: privateSubnetsForFsx.map(s => s.routeTable.routeTableId),
                throughputCapacity: 256,
            },
            securityGroupIds: [fsxSecurityGroup.securityGroupId],
            storageCapacity: 10240,
            storageType: 'SSD',
        });
        const svm = new aws_cdk_lib_1.aws_fsx.CfnStorageVirtualMachine(this, `${name}-svm`, {
            name,
            fileSystemId: cfnFileSystem.ref,
            rootVolumeSecurityStyle: 'MIXED',
        });
        new aws_cdk_lib_1.aws_fsx.CfnVolume(this, `${name}-volume`, {
            /**
             * A volume name must begin with a letter or underscore,
             * can only contain alphanumeric characters or underscores (`_`),
             * and cannot exceed 203 characters in length
             */
            name: this.trimStringAt203rdCharacter(this.removeNonAlphanumericOrUnderscores(this.replaceDashesWithUnderscores(name)).concat('volume')),
            volumeType: 'ONTAP',
            ontapConfiguration: {
                junctionPath: this.mountName,
                securityStyle: 'MIXED',
                sizeInMegabytes: '102400',
                storageVirtualMachineId: svm.ref,
                storageEfficiencyEnabled: 'true',
            },
        });
        this.dnsName = `${svm.logicalId}.${cfnFileSystem.ref}.fsx.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`;
        new aws_cdk_lib_1.CfnOutput(this, `${name}-fsx-admin-password-name`, {
            value: fsxAdminPassword.secretName,
        });
    }
    replaceDashesWithUnderscores(str) {
        return str.replace(/-/g, '_');
    }
    removeNonAlphanumericOrUnderscores(str) {
        return str.replace(/[^a-zA-Z0-9_]/g, '');
    }
    trimStringAt203rdCharacter(str) {
        return str.substring(0, 202);
    }
    /**
    * Configure Security Group for FsX
    * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/limit-access-security-groups.html
    */
    addIngressToFsxFromSourceSg(securityGroupSource, fsxSecurityGroup) {
        fsxSecurityGroup.addIngressRule(aws_ec2_1.Peer.anyIpv4(), aws_ec2_1.Port.tcp(2049), 'Allow 2049 inbound from anywhere');
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.icmpPing());
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(22));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(111));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(135));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(139));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(161));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(162));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(443));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(445));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(635));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(749));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(2049));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(3260));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(4045));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(4046));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(11104));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.tcp(11105));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(111));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(135));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(137));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(139));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(161));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(162));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(635));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(2049));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(4045));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(4046));
        fsxSecurityGroup.addIngressRule(securityGroupSource, aws_ec2_1.Port.udp(4049));
    }
}
_a = JSII_RTTI_SYMBOL_1;
FsxOntap[_a] = { fqn: "cdk-fsx-ontap.FsxOntap", version: "1.0.12" };
exports.FsxOntap = FsxOntap;
//# sourceMappingURL=data:application/json;base64,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