import csv
import os.path
import shutil
from string import Template
from typing import Dict, List

from black import FileMode, format_str

configFile = "block_config.csv"
savePath = "../../nodedge/blocks/autogen/"
initFilename = "__init__.py"


def _prepend_socket_type(types_string):
    types_string = types_string[1:-1]  # remove parenthesis
    types_list = types_string.split(",")
    types_string = (
        "[" + ", ".join(["SocketType." + currType for currType in types_list]) + ",]"
    )

    return types_string


def _generate_init_files(libraryDict):
    # Create init file
    initFileString = ""
    for lib in sorted(libraryDict.keys()):
        initFileString += f"from .{lib} import *\n"
    initFilePath = os.path.join(savePath, initFilename)
    initFile = open(initFilePath, "w")
    initFileString = format_str(initFileString, mode=FileMode())
    initFile.write(initFileString)
    initFile.close()

    # Create init file for each autogenerated library
    initFileString = ""
    for lib, blockList in libraryDict.items():
        for blockName in sorted(blockList):
            initFileString += f"from .{blockName}_block import *\n"
        initFilePath = os.path.join(savePath, lib, initFilename)
        initFile = open(initFilePath, "w")
        initFileString = format_str(initFileString, mode=FileMode())
        initFile.write(initFileString)
        initFile.close()


def main():
    libraries: Dict[str, List[str]] = {}
    # Create `autogen` folder for generated block libraries
    if not os.path.exists(savePath):
        os.makedirs(savePath)
    else:
        for filename in os.listdir(savePath):
            filepath = os.path.join(savePath, filename)
            for filename in os.listdir(savePath):
                filepath = os.path.join(savePath, filename)
                try:
                    shutil.rmtree(filepath)
                except OSError:
                    os.remove(filepath)
    with open(configFile) as infile:
        reader = csv.DictReader(infile, delimiter=";")

        # Read one line (corresponding to one block) from the csv
        for row in reader:

            # Save block in dictionary
            if row["library"] not in libraries.keys():
                libraries[row["library"]] = [row["function"]]
            else:
                libraries[row["library"]].append(row["function"])

            # Add socket type object
            row["input_socket_types"] = _prepend_socket_type(row["input_socket_types"])
            row["output_socket_types"] = _prepend_socket_type(
                row["output_socket_types"]
            )

            # Generate current block
            with open("block_template.txt") as templateFile:
                inputData = templateFile.read()

            template = Template(inputData)

            outputData = template.substitute(**row)
            outputData = format_str(outputData, mode=FileMode())
            filename = f"{(row['function'])}_block.py"
            libraryPath = os.path.join(savePath, row["library"])
            if not os.path.exists(libraryPath):
                os.makedirs(libraryPath)
            filePath = os.path.join(libraryPath, filename)
            outputFile = open(filePath, "w")
            outputFile.write(outputData)
            outputFile.close()

        templateFile.close()
    infile.close()
    _generate_init_files(libraries)


if __name__ == "__main__":
    main()

# TODO: Generate test for each block in a separated file
