# MPWF



Based on the testdriven.io tutorial "Building your own Python web framework".

![purpose](https://img.shields.io/badge/purpose-education-green.svg)


It's a WSGI framework and can be used with any WSGI application server such as Gunicorn.

*This documentation is an adapted copy of the original one from the tutorial.*

## Installation

```shell
pip install mpwf
```

## How to use it

### Basic usage:

```python
from mpwf.api import API

app = API()

# Simple route
@app.route("/home")
def home(request, response):
    response.text = "Hello from the HOME page"

# Parametrized 
@app.route("/hello/{name}")
def greeting(request, response, name):
    response.text = f"Hello, {name}"

# Class Based 
@app.route("/book")
class BooksResource:
    def get(self, req, resp):
        resp.text = "Books Page"

    def post(self, req, resp):
        resp.text = "Endpoint to create a book"

# With Template  
@app.route("/templates")
def template_handler(req, resp):
    resp.html = app.template("index.html", context={"name": "MPWF", "title": "A title"})

# JSON  
@app.route("/templates")
def template_handler(req, resp):
    resp.json = {"name": "data", "type": "JSON"}
```

## Templates

The default folder for templates is `templates`. You can change it when initializing the main `API()` class:

```python
app = API(templates_dir="templates_dir_name")
```

Then you can use HTML files in that folder like so in a handler:

```python
@app.route("/show/templates")
def handler_with_template(req, resp):
    resp.html = app.template(
        "example.html", context={"title": "Awesome Framework", "body": "welcome to the future!"})
```

## Static Files

Just like templates, the default folder for static files is `static` and you can override it:

```python
app = API(static_dir="static_dir_name")
```

Then you can use the files inside this folder in HTML files:

```html
<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="UTF-8">
  <title>{{title}}</title>

  <link href="/app/static/main.css" rel="stylesheet" type="text/css">
</head>

<body>
    <h1>{{body}}</h1>
    <p>This is a paragraph</p>
</body>
</html>
```

### Middleware

You can create custom middleware classes by inheriting from the `bumbo.middleware.Middleware` class and overriding its two methods
that are called before and after each request:

```python

# [...]
class SimpleCustomMiddleware(Middleware):
    def process_request(self, req):
        print("Before dispatch", req.url)

    def process_response(self, req, res):
        print("After dispatch", req.url)


app.add_middleware(SimpleCustomMiddleware)
```

### Custom Exception Handling

You can create custom exception handlers like so:

```python
def custom_exception_handler(request, response, exception_cls):
    if request.method == 'GET':
        response.text = f"Error: {str(exception_cls)}"

# Then add it to the framework: 
app.add_exception_handler(custom_exception_handler) 
```