"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MultipartUserData = exports.MultipartBody = exports.UserData = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts
     */
    static forLinux(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_LinuxUserDataOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.forLinux);
            }
            throw error;
        }
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    static forOperatingSystem(os) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_OperatingSystemType(os);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.forOperatingSystem);
            }
            throw error;
        }
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
_a = JSII_RTTI_SYMBOL_1;
UserData[_a] = { fqn: "@aws-cdk/aws-ec2.UserData", version: "1.157.0" };
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        var _d;
        const shebang = (_d = this.props.shebang) !== null && _d !== void 0 ? _d : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'` + (params.region !== undefined ? ` --region ${params.region}` : ''));
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = resource.node.defaultChild.logicalId;
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop` + (params.region !== undefined ? ` -Region ${params.region}` : ''));
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands(`&'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = resource.node.defaultChild.logicalId;
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
/**
 * The base class for all classes which can be used as {@link MultipartUserData}.
 */
class MultipartBody {
    constructor() {
    }
    /**
     * Constructs the new `MultipartBody` wrapping existing `UserData`. Modification to `UserData` are reflected
     * in subsequent renders of the part.
     *
     * For more information about content types see {@link MultipartBodyOptions.contentType}.
     *
     * @param userData user data to wrap into body part
     * @param contentType optional content type, if default one should not be used
     */
    static fromUserData(userData, contentType) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_UserData(userData);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromUserData);
            }
            throw error;
        }
        return new MultipartBodyUserDataWrapper(userData, contentType);
    }
    /**
     * Constructs the raw `MultipartBody` using specified body, content type and transfer encoding.
     *
     * When transfer encoding is specified (typically as Base64), it's caller responsibility to convert body to
     * Base64 either by wrapping with `Fn.base64` or by converting it by other converters.
     */
    static fromRawBody(opts) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_MultipartBodyOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromRawBody);
            }
            throw error;
        }
        return new MultipartBodyRaw(opts);
    }
}
exports.MultipartBody = MultipartBody;
_b = JSII_RTTI_SYMBOL_1;
MultipartBody[_b] = { fqn: "@aws-cdk/aws-ec2.MultipartBody", version: "1.157.0" };
/**
 * Content type for shell scripts
 */
MultipartBody.SHELL_SCRIPT = 'text/x-shellscript; charset="utf-8"';
/**
 * Content type for boot hooks
 */
MultipartBody.CLOUD_BOOTHOOK = 'text/cloud-boothook; charset="utf-8"';
/**
 * The raw part of multi-part user data, which can be added to {@link MultipartUserData}.
 */
class MultipartBodyRaw extends MultipartBody {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.props.contentType}`);
        if (this.props.transferEncoding != null) {
            result.push(`Content-Transfer-Encoding: ${this.props.transferEncoding}`);
        }
        // One line free after separator
        result.push('');
        if (this.props.body != null) {
            result.push(this.props.body);
            // The new line added after join will be consumed by encapsulating or closing boundary
        }
        return result;
    }
}
/**
 * Wrapper for `UserData`.
 */
class MultipartBodyUserDataWrapper extends MultipartBody {
    constructor(userData, contentType) {
        super();
        this.userData = userData;
        this.contentType = contentType || MultipartBody.SHELL_SCRIPT;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.contentType}`);
        result.push('Content-Transfer-Encoding: base64');
        result.push('');
        result.push(core_1.Fn.base64(this.userData.render()));
        return result;
    }
}
/**
 * Mime multipart user data.
 *
 * This class represents MIME multipart user data, as described in.
 * [Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)
 *
 */
class MultipartUserData extends UserData {
    constructor(opts) {
        super();
        this.parts = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_MultipartUserDataOptions(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        let partsSeparator;
        // Validate separator
        if ((opts === null || opts === void 0 ? void 0 : opts.partsSeparator) != null) {
            if (new RegExp(MultipartUserData.BOUNDRY_PATTERN).test(opts.partsSeparator)) {
                throw new Error(`Invalid characters in separator. Separator has to match pattern ${MultipartUserData.BOUNDRY_PATTERN}`);
            }
            else {
                partsSeparator = opts.partsSeparator;
            }
        }
        else {
            partsSeparator = '+AWS+CDK+User+Data+Separator==';
        }
        this.opts = {
            partsSeparator: partsSeparator,
        };
    }
    /**
     * Adds a part to the list of parts.
     */
    addPart(part) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_MultipartBody(part);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPart);
            }
            throw error;
        }
        this.parts.push(part);
    }
    /**
     * Adds a multipart part based on a UserData object.
     *
     * If `makeDefault` is true, then the UserData added by this method
     * will also be the target of calls to the `add*Command` methods on
     * this MultipartUserData object.
     *
     * If `makeDefault` is false, then this is the same as calling:
     *
     * ```ts
     * declare const multiPart: ec2.MultipartUserData;
     * declare const userData: ec2.UserData;
     * declare const contentType: string;
     *
     * multiPart.addPart(ec2.MultipartBody.fromUserData(userData, contentType));
     * ```
     *
     * An undefined `makeDefault` defaults to either:
     * - `true` if no default UserData has been set yet; or
     * - `false` if there is no default UserData set.
     */
    addUserDataPart(userData, contentType, makeDefault) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_UserData(userData);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addUserDataPart);
            }
            throw error;
        }
        this.addPart(MultipartBody.fromUserData(userData, contentType));
        makeDefault = makeDefault !== null && makeDefault !== void 0 ? makeDefault : (this.defaultUserData === undefined ? true : false);
        if (makeDefault) {
            this.defaultUserData = userData;
        }
    }
    render() {
        const boundary = this.opts.partsSeparator;
        // Now build final MIME archive - there are few changes from MIME message which are accepted by cloud-init:
        // - MIME RFC uses CRLF to separate lines - cloud-init is fine with LF \n only
        // Note: new lines matters, matters a lot.
        var resultArchive = new Array();
        resultArchive.push(`Content-Type: multipart/mixed; boundary="${boundary}"`);
        resultArchive.push('MIME-Version: 1.0');
        // Add new line, the next one will be boundary (encapsulating or closing)
        // so this line will count into it.
        resultArchive.push('');
        // Add parts - each part starts with boundary
        this.parts.forEach(part => {
            resultArchive.push(`--${boundary}`);
            resultArchive.push(...part.renderBodyPart());
        });
        // Add closing boundary
        resultArchive.push(`--${boundary}--`);
        resultArchive.push(''); // Force new line at the end
        return resultArchive.join('\n');
    }
    addS3DownloadCommand(params) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_S3DownloadOptions(params);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addS3DownloadCommand);
            }
            throw error;
        }
        if (this.defaultUserData) {
            return this.defaultUserData.addS3DownloadCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addExecuteFileCommand(params) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_ExecuteFileOptions(params);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addExecuteFileCommand);
            }
            throw error;
        }
        if (this.defaultUserData) {
            this.defaultUserData.addExecuteFileCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addSignalOnExitCommand(resource) {
        if (this.defaultUserData) {
            this.defaultUserData.addSignalOnExitCommand(resource);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addOnExitCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addOnExitCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
}
exports.MultipartUserData = MultipartUserData;
_c = JSII_RTTI_SYMBOL_1;
MultipartUserData[_c] = { fqn: "@aws-cdk/aws-ec2.MultipartUserData", version: "1.157.0" };
MultipartUserData.USE_PART_ERROR = 'MultipartUserData only supports this operation if it has a default UserData. Call addUserDataPart with makeDefault=true.';
MultipartUserData.BOUNDRY_PATTERN = '[^a-zA-Z0-9()+,-./:=?]';
//# sourceMappingURL=data:application/json;base64,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