# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rnc']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=0.7.0,<0.8.0',
 'aiohttp>=3.7.4.post0,<3.8.0',
 'beautifulsoup4>=4.9.3,<4.10.0',
 'bs4>=0.0.1,<0.1.0',
 'lxml>=4.6.3,<4.7.0',
 'ujson>=4.1.0,<4.2.0']

setup_kwargs = {
    'name': 'rnc',
    'version': '0.7.0',
    'description': 'API for Russian National Corpus',
    'long_description': "# API for [Russian National Corpus](http://ruscorpora.ru) \n\n## Installation\n```bash\npip install rnc\n```\n\n## Structure\nCorpus object contains list of obtained examples.\nThere are two types of examples:\n![](https://github.com/kunansy/RNC/blob/master/docs/Two_ex_types.png?raw=true) <br> \n* If `out` is `normal`, API uses normal example, which name is equal to the Corpus class name:\n\n```python\nru = rnc.MainCorpus(...)\nru.request_examples()\n\nprint(type(ru[0]))\n>>> MainExample\n```\n* if `out` is `kwic`, API uses `KwicExample`.\n\nExamples' objects [fields](https://github.com/kunansy/RNC/blob/master/docs/Examples.md)   \n\n\n## Usage\n```python\nimport rnc\n\nru = rnc.MainCorpus(\n    query='корпус', \n    p_count=5,\n    file='filename.csv',\n    marker=str.upper,\n    **kwargs\n)\n\nru.request_examples()\n```\n* `query` – one str or dict with tags. Words to find, you should give the vocabulary form of them.\n* `p_count` – count of **PAGES**.\n* `file` – path to local csv file, optional. Example: `file='data\\\\filename.csv'`. \n* `marker` – function, with which found wordforms will be marked, optional. \n* `kwargs` – additional params.\n\n[Corpora](https://github.com/kunansy/RNC/blob/master/docs/Corpora.md) you can use.\n\n\n### Full query form\n```python\nquery = {\n    'word1': {\n        'gramm': 'acc', # grammar tags for lexgramm search\n        'flags': 'bdot' # additional tags for lexgramm search\n    },\n    # you can get as a value one string or dict of params\n    # params are: any name of dict key, name of tag (you can see them below)  \n    'word2': {\n        'gramm': { \n            # the NAMES of these keys might be any\n            'pos (any name)': 'S' or ['S', 'A'], # one value or list of values,\n            'case (any name)': 'acc' or ['acc', 'nom'],\n        },\n        'flags': {}, # all the same to here\n        # distance between first and second words\n        'min': 1,  \n        'max': 3\n    },  \n}\n\ncorp = rnc.MainCorpus(\n    query, 5, file='filename.csv', marker=str.upper, **kwargs)\ncorp.reques_examples()\n```\n[Lexgramm search params](https://github.com/kunansy/RNC/tree/master/docs/Lexgram%20search%20params)\n\n\n### String as a query\nAlso you can pass as a query a string with the **vocabulary forms** of the \nwords, divided by space: `query = 'get down'` or `query = 'я получить'`. \nDistance between them will be default.\n\n\n### Additional request params\nThese params are optional, you can ignore them. Here are the default values.\n```python\ncorp = rnc.ParallelCorpus(\n    query=query, \n    p_count=5,\n    file='filename.csv',\n    marker=str.upper,\n    \n    dpp=5, # documents per page\n    spd=10, # sentences per document (<= than spd)\n    text='lexgramm' or 'lexform', # way to search\n    out='normal' or 'kwic', # output format\n    kwsz=5, # if out=kwic, count of words in context\n    sort='i_grtagging', # way to sort the results, see HOWTO section below\n    mycorp='', # see HOWTO section below\n    accent=0, # with accentology (1) or without (0), if it is available\n)\n```\n[Sort keys](https://github.com/kunansy/RNC/blob/master/docs/HTTP%20params.md)\n\n\n### API can work with a local file too\n```python\nru = rnc.SpokenCorpus(file='local_database.csv') # it must exist\nprint(ru)\n```\nIf the file exists, API works with it. If the data list is not empty you \ncannot request new examples. <br>\n\nIf you work with a file, it is not demanded to pass any argument to Corpus \nexcept for the file name (`file=...`).\n\n\n### Working with corpora\n```python\ncorp = rnc.corpus_name(...) \n```\n* `corp.request_examples()` – request examples. \nThere is an exception if:\n    * Data still exist. \n    * No results found.\n    * A requested page does not exist (if there are 10 pages in the RNC, but \n      you have requested > 10).\n    * There is a mistake in the request.\n    * You have no access to the Internet.\n    * There is a problem while getting access to RNC.\n    * another problems...\n* `corp.data` – list of examples (only getter)\n* `corp.query` – query (only getter).\n* `corp.forms_in_query` – requested wordforms (only getter).\n* `corp.p_count` – requested count of pages (only getter). \n* `corp.file` – path to the local csv file (only getter).\n* `corp.marker` – marker (only getter).\n* `corp.params` – dict, HTTP tags (only getter). \n* `corp.found_wordforms` – dict with found wordforms and their frequency (only getter).\n* `corp.ex_type` – type of example (only getter).\n* `corp.amount_of_docs` – amount of docs where the query was found.\n* `corp.amount_of_contexts` – amount of contexts where the query was found.\n* `corp.graphic_link` – link to the graphic of the distribution of query occurrences by years.\n* `corp.dump()` – write two files: csv file with all data and json file with config.\n* `corp.copy()` – create a copy.\n* `corp.shuffle()` – shuffle data list.\n* `corp.sort_data(key=, reverse=)` – sort the list of examples. Here HTTP keys do not work,\nkey is applied to Example objects.  \n* `corp.pop(index)` – remove and return the example at the index.\n* `corp.clear()` – empty the data list.\n* `corp.filter(key)` – filter the data list, remove some examples using the key. \nKey is applied to the `Example` objects.\n* `corp.url` – URL of the first RNC page (only getter).\n* `corp.findall(pattern, args)` – get all examples where the pattern found and \n  the match.\n* `corp.finditer(pattern, args)` – get all examples where the pattern found and \n  the match.\n\nMagic methods: \n* `corp.dpp` or another request param (only getter).\n* `corp()` – all the same to `request_examples()`.\n* `str(corp) or print(corp)` – str with info about Corpus, enumerated examples.\nBy default, Corpus shows first 50 examples, but you can change it \nor turn the restriction off. \n\n    Info about Corpus:\n    ```\n    Russian National Corpus (https://ruscorpora.ru)\n    Class: CorpusName, len = amount of examples \n    Pages: n of 'words' requested\n    ```\n* `len(corp)` – count of examples.\n* `bool(corp)` – whether data exist.\n* `corp[index or slice]` – get element at the index or create a new object \n  with sliced data:\n```python\nfrom_2_to_10 = corp[2:10:2]\n```\n* `del corp[10]` or `del corp[:10]` – remove some examples from the data list.\n\n* Also you can use cycle `for`. For example we want to see only left \n  context (`out=kwic`) and source:\n```python\ncorp = rnc.ParallelCorpus(\n    'corpus', 5, \n    out='kwic', kwsz=7, \n    mycorp=rnc.mycorp.en\n)\ncorp.request_examples()\n\nfor r in corp:\n    print(r.left)\n    print(r.src)\n```\n\nSet default values to all objects you will create:\n* `corpus_name.set_dpp(value)` – change default `document per page` value.\n* `corpus_name.set_spd(value)` – change default `sentences per document` value.\n* `corpus_name.set_text(value)` – change default search way.\n* `corpus_name.set_sort(value)` – change default sort key.\n* `corpus_name.set_min(value)` – change default min distance between words.\n* `corpus_name.set_max(value)` – change default max distance between words.\n* `corpus_name.set_restrict_show(value)` – change default amount of shown examples in print. \nIf it is equal to `False`, the Corpus shows all examples. \n\n\n### Corpora features\n#### ParallelCorpus\n* The query might be both in the original language and in the language of \n  translation. \n\n#### MultilingualParaCorpus\n* Working with files is removed.\n* Param `mycorp` is not demanded by default, but it might be passed, see \n  **HOWTO** section below.\n\n#### MultimodalCorpus\n* `corp.download_all()` – download all media files. **It is recommended** to use \nthis method instead of `expl.download_file()`.\n\n\n## Logger\n* See all log messages\n```python\nrnc.set_stream_handler_level('debug')\n```\n* See less than all messages\n```python\nrnc.set_stream_handler_level('info')\n```\n* Turn the logger off\n```python\nrnc.set_logger_level('critical')\n```\n* Turn off all messages in the stream, but dump logs to file\n```python\nrnc.set_stream_handler_level('critical')\n```\n* Turn off dumping logs to file\n```python\nrnc.set_file_handler_level('critical')\n```\n\n\n## ATTENTION\n* Do not forget to call this function\n```python\ncorp.request_examples()\n```\n* If you have requested more than 10 pages, RNC returns 429 error \n  (Too many requests).\nFor example requesting 100 pages you should wait about 3 minutes: \n![100 pages](https://github.com/kunansy/RNC/blob/master/docs/100_pages.png?raw=true)\n* **Do not call** the marker you pass\n\n**RIGHT:**\n```python\nru = rnc.MainCorpus(...,  marker=str.upper)\n```\n**WRONG:**\n```python\nru = rnc.MainCorpus(..., marker=str.upper())\n```\n* Pass an empty string as a param if you do not want to set them\n```python\nquery = {\n    'word1': '',\n    'word2': {'min': 2, 'max': 5}\n}\n```\n* If `accent=1`, marker does not work.\n\n---\n\n## HOWTO\nYou can ask any question you want [here](https://github.com/kunansy/RNC/discussions).\n\n### How to set sort?\nThere are some sort keys:\n1. `i_grtagging` – by default.\n2. `random` – randomly.\n3. `i_grauthor` – by author.\n4. `i_grcreated_inv` – by creation date.\n5. `i_grcreated` – by creation date in reversed order.\n6. `i_grbirthday_inv` – by author's birth date.\n7. `i_grbirthday` – by author's birth date in reversed order.\n\n[Some of HTTP params](https://github.com/kunansy/RNC/blob/master/docs/HTTP%20params.md).\n\n\n### How to set language in ParallelCorpus?\n```python\nen = rnc.ParallelCorpus('get', 5, mycorp=rnc.mycorp.en)\n```\n**OR**\n```python\nen = rnc.ParallelCorpus('get', 5, mycorp=rnc.mycorp['en'])\n```\nLanguage keys list:\n1. Armenian – 'arm'\n1. Bashkir – 'bas'\n1. Belarusian – 'bel'\n1. Bulgarian – 'bul'\n1. Buryatian – 'bur'\n1. Chinese – 'ch'\n1. Czech – 'cz'\n1. English – 'en'\n1. Estonian – 'es'\n1. Finnish – 'fin'\n1. French – 'fr'\n1. German – 'ger'\n1. Italian – 'it'\n1. Latvian – 'lat'\n1. Lithuanian – 'lit'\n1. Polish – 'pol'\n1. Spanish – 'sp'\n1. Swedish – 'sw'\n1. Ukrainian – 'ukr'\n\nIf you want to search something by several languages, choose and set the \n`mycorp` in the site, pass this param to Corpus. \n\n\n### How to set subcorpus?\nMeans specify the sample where you want to search the query. <br>\n\nThere are default keys in `rnc.mycorp` (working checked in \n**MainCorpus**) – Russian writers and poets: \n* Pushkin\n* Dostoyevsky\n* TolstoyLN\n* Chekhov\n* Gogol\n* Turgenev\n\nExample:\n```python\nru = rnc.MainCorpus('нету', 1, mycorp=rnc.mycorp['Pushkin'])\n```\n\n\n**OR**\n\n```python\nru = rnc.MainCorpus('нету', 1, mycorp=rnc.mycorp.Pushkin)\n```\n\n\n**OR**\n\n \n![1](https://raw.githubusercontent.com/kunansy/RNC/master/docs/How%20to%20set%20subcorpus/1.png)\n![2](https://raw.githubusercontent.com/kunansy/RNC/master/docs/How%20to%20set%20subcorpus/2.png)\n![3](https://raw.githubusercontent.com/kunansy/RNC/master/docs/How%20to%20set%20subcorpus/3.png)\n![4](https://raw.githubusercontent.com/kunansy/RNC/master/docs/How%20to%20set%20subcorpus/4.png)\n\n\n## Links\n* [Russian National Corpus](https://ruscorpora.ru)\n* [Docs](https://github.com/kunansy/RNC/tree/master/docs)\n* Examples' objects [fields](https://github.com/kunansy/RNC/blob/master/docs/Examples.md)\n* [Corpora](https://github.com/kunansy/RNC/blob/master/docs/Corpora.md) you can use.\n* [Lexgramm search params](https://github.com/kunansy/RNC/tree/master/docs/Lexgram%20search%20params)\n* [Sort keys](https://github.com/kunansy/RNC/blob/master/docs/HTTP%20params.md)\n---\n\n\n## Requirements\n* Python >= 3.7\n\n\n## Licence\n`rnc` is offered under MIT licence.\n\n\n## Source code\nThe project is hosted on [Github](https://github.com/kunansy/RNC)\n\n---\n\nPlease file an issue in the [bug tracker](https://github.com/kunansy/RNC/issues) \nif you have found a bug or have some suggestions to improve the library.\n",
    'author': 'kunansy',
    'author_email': 'kolobov.kirill@list.ru',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://kunansy.github.io/RNC/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
