"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricSet = exports.allMetricsGraphJson = void 0;
const drop_empty_object_at_the_end_of_an_array_token_1 = require("./drop-empty-object-at-the-end-of-an-array-token");
const env_tokens_1 = require("./env-tokens");
const metric_util_1 = require("./metric-util");
const object_1 = require("./object");
/**
 * Return the JSON structure which represents these metrics in a graph.
 *
 * Depending on the metric type (stat or expression), one `Metric` object
 * can render to multiple time series.
 *
 * - Top-level metrics will be rendered visibly, additionally added metrics will
 *   be rendered invisibly.
 * - IDs used in math expressions need to be either globally unique, or refer to the same
 *   metric object.
 *
 * This will be called by GraphWidget, no need for clients to call this.
 */
function allMetricsGraphJson(left, right) {
    // Add metrics to a set which will automatically expand them recursively,
    // making sure to retain conflicting the visible one on conflicting metrics objects.
    const mset = new MetricSet();
    mset.addTopLevel('left', ...left);
    mset.addTopLevel('right', ...right);
    // Render all metrics from the set.
    return mset.entries.map(entry => new drop_empty_object_at_the_end_of_an_array_token_1.DropEmptyObjectAtTheEndOfAnArray(metricGraphJson(entry.metric, entry.tag, entry.id)));
}
exports.allMetricsGraphJson = allMetricsGraphJson;
function metricGraphJson(metric, yAxis, id) {
    const config = metric.toMetricConfig();
    const ret = [];
    const options = { ...config.renderingProperties };
    metric_util_1.dispatchMetric(metric, {
        withStat(stat) {
            ret.push(stat.namespace, stat.metricName);
            // Dimensions
            for (const dim of (stat.dimensions || [])) {
                ret.push(dim.name, dim.value);
            }
            // Metric attributes that are rendered to graph options
            if (stat.account) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(stat.account);
            }
            if (stat.region) {
                options.region = env_tokens_1.regionIfDifferentFromStack(stat.region);
            }
            if (stat.period && stat.period.toSeconds() !== 300) {
                options.period = stat.period.toSeconds();
            }
            if (stat.statistic && stat.statistic !== 'Average') {
                options.stat = stat.statistic;
            }
        },
        withExpression(expr) {
            options.expression = expr.expression;
            if (expr.searchAccount) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(expr.searchAccount);
            }
            if (expr.searchRegion) {
                options.region = env_tokens_1.regionIfDifferentFromStack(expr.searchRegion);
            }
            if (expr.period && expr.period !== 300) {
                options.period = expr.period;
            }
        },
    });
    // Options
    if (!yAxis) {
        options.visible = false;
    }
    if (yAxis !== 'left') {
        options.yAxis = yAxis;
    }
    if (id) {
        options.id = id;
    }
    // If math expressions don't have a label (or an ID), they'll render with an unelegant
    // autogenerated id ("metric_alias0"). Our ids may in the future also be autogenerated,
    // so if an ME doesn't have a label, use its toString() as the label (renders the expression).
    if (options.visible !== false && options.expression && !options.label) {
        options.label = metric.toString();
    }
    const renderedOpts = object_1.dropUndefined(options);
    if (Object.keys(renderedOpts).length !== 0) {
        ret.push(renderedOpts);
    }
    return ret;
}
/**
 * Contain a set of metrics, expanding math expressions
 *
 * "Primary" metrics (added via a top-level call) can be tagged with an additional value.
 */
class MetricSet {
    constructor() {
        this.metrics = new Array();
        this.metricById = new Map();
        this.metricByKey = new Map();
    }
    /**
     * Add the given set of metrics to this set
     */
    addTopLevel(tag, ...metrics) {
        for (const metric of metrics) {
            this.addOne(metric, tag);
        }
    }
    /**
     * Access all the accumulated timeseries entries
     */
    get entries() {
        return this.metrics;
    }
    /**
     * Add a metric into the set
     *
     * The id may not be the same as a previous metric added, unless it's the same metric.
     *
     * It can be made visible, in which case the new "metric" object replaces the old
     * one (and the new ones "renderingPropertieS" will be honored instead of the old
     * one's).
     */
    addOne(metric, tag, id) {
        const key = metric_util_1.metricKey(metric);
        let existingEntry;
        // Try lookup existing by id if we have one
        if (id) {
            existingEntry = this.metricById.get(id);
            if (existingEntry && metric_util_1.metricKey(existingEntry.metric) !== key) {
                throw new Error(`Cannot have two different metrics share the same id ('${id}') in one Alarm or Graph. Rename one of them.`);
            }
        }
        if (!existingEntry) {
            // Try lookup by metric if we didn't find one by id
            existingEntry = this.metricByKey.get(key);
            // If the one we found already has an id, it must be different from the id
            // we're trying to add and we want to add a new metric. Pretend we didn't
            // find one.
            if ((existingEntry === null || existingEntry === void 0 ? void 0 : existingEntry.id) && id) {
                existingEntry = undefined;
            }
        }
        // Create a new entry if we didn't find one so far
        let entry;
        if (existingEntry) {
            entry = existingEntry;
        }
        else {
            entry = { metric };
            this.metrics.push(entry);
            this.metricByKey.set(key, entry);
        }
        // If it didn't have an id but now we do, add one
        if (!entry.id && id) {
            entry.id = id;
            this.metricById.set(id, entry);
        }
        // If it didn't have a tag but now we do, add one
        if (!entry.tag && tag) {
            entry.tag = tag;
        }
        // Recurse and add children
        const conf = metric.toMetricConfig();
        if (conf.mathExpression) {
            for (const [subId, subMetric] of Object.entries(conf.mathExpression.usingMetrics)) {
                this.addOne(subMetric, undefined, subId);
            }
        }
    }
}
exports.MetricSet = MetricSet;
//# sourceMappingURL=data:application/json;base64,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