# -*- coding: utf-8 -*-
import os.path

import numpy as np
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
import time
from datetime import datetime
import multiprocessing
import string
from rich.console import Console
try:
    from rich.console import Group
except:
    from rich.console import RenderGroup as Group    
from rich.progress import Progress, BarColumn, TimeRemainingColumn, TimeElapsedColumn
from rich.live import Live
from rich.table import Table
import indago


class Optimizer:
    """Base class for all optimization methods.

    Parameters
    ----------
    dimensions: int
        Number of dimensions of the search space i.e. number of optimization variables.
    evaluation_function: callable
        A function (or a callable class) which takes a Numpy:ndarray as argument and returns a Numpy:ndarray containing
        objectives' and constraint's values. Optionally, it can take a keyword argument :code:`s` in order to handle
        unique string generated by the optimizer.
    monitoring: str or None
        Adjust the monitoring style and level of output produced during the optimization. :code:`monitoring='None'` or
        :code:`monitoring=None` suppress all output. For displaying a per iteration output to the terminal use
        :code:`monitoring='basic'` which can display a colored output if supported by your terminal. For displaying a
         progress bar and dynamically updated best candidate state use :code:`monitoring='dashboard'`.
    best: CandidateState
        The best candidate state encountered in the optimization process.
    """

    def __init__(self):
        """Initialization for Optimizer class defines member variables and set initial values.

        Returns
        -------
        Optimizer
            Optimizer instance
        """

        self.dimensions = None
        self.evaluation_function = None
        self.number_of_processes = 1
        
        self.objectives = 1
        self.objective_weights = None
        self.objective_labels = None
        self.constraints = 0
        self.constraint_labels = None
        
        self.iterations = None        
        self.maximum_evaluations = None        
        self.maximum_stalled_iterations = None
        self.maximum_stalled_evaluations = None       
        self.target_fitness = None
        
        self.lb = None
        self.ub = None
        self.best = None
        self.X0 = None

        self.history = None
        self.monitoring = None
        self._progress_log = None
        self.convergence_log_file = None
        self._time1 = time.time()

        self.forward_unique_str = False
        self._unique_str_list = []
        self.post_iteration_processing = None
        
        self.safe_evaluation = None
        self.eval_fail_count = 0
        self.eval_fail_behavior = None
        self.eval_retry_attempts = None
        self.eval_retry_recede = None
        
        self.it = 0
        self.eval = 0
        self._stalled_it = None
        self._stalled_eval = None

        self.params = {}
        
    def _progress_factor(self):
        """A private method for calculating progress factor ranging from zero to one,
        based on iterations or evaluations (which ever is running out faster).
        
        Returns
        -------
        progress factor: float
            A number between zero and one, representing the progress of the optimization process.
        """
        prog = [self.it / self.iterations]
        if self.maximum_evaluations:
            prog.append(self.eval / self.maximum_evaluations)
        return max(prog)  
    
    def _evaluation_function_safe(self, X, s=None):
        """A private method for wrapping evaluation function in try-except.

        Returns
        -------
        fitness: float
            Fitness value, or np.nan if failed to compute evaluation function.
        """
        try:
            if s is None:
                return self.evaluation_function(X)
            else:
                return self.evaluation_function(X, s)
        except:
            return np.nan
    
    def _init_optimizer(self):
        """A private method for Optimizer initialization which checks the types and values of Optimizer member
        parameters. Automatically sets optional values if they are not provided.

        Returns
        -------
        None
            Nothing
        """
        
        assert isinstance(self.dimensions, int) and self.dimensions > 0, \
            "optimizer.dimensions should be positive integer"
            
        if np.size(self.lb) == 1 and np.size(self.ub) == 1:
            self.lb = np.full(self.dimensions, self.lb)
            self.ub = np.full(self.dimensions, self.ub)
        # in case lb/ub are lists or tuples
        self.lb = np.array(self.lb)
        self.ub = np.array(self.ub)
            
        assert np.size(self.lb) == self.dimensions and \
            np.size(self.ub) == self.dimensions, \
            "optimizer.lb and optimizer.ub should be of size optimizer.dimensions"
        
        if self.X0 is not None:
            assert isinstance(self.X0, np.ndarray), \
                "optimizer.X0 should be 1d or 2d np.array"
            assert self.X0.ndim == 1 or self.X0.ndim == 2, \
                "optimizer.X0 should be 1d or 2d np.array"
            # if 1D convert to 2D
            if self.X0.ndim == 1:
                self.X0 = np.array([self.X0])
        
        assert callable(self.evaluation_function), \
            "optimizer.evaluation_function should be callable"
        
        if self.safe_evaluation is None:
            self.safe_evaluation = False
        assert isinstance(self.safe_evaluation, bool), \
            "optimizer.safe_evaluation should be True/False"
        if not self.safe_evaluation:
            self._evaluation_function_safe = self.evaluation_function

        assert isinstance(self.objectives, int) and self.objectives > 0, \
            "optimizer.objectives should be positive integer"
        
        assert isinstance(self.constraints, int) and self.constraints >= 0, \
            "optimizer.constraints should be non-negative integer"
        
        if self.objective_weights is None:
            self.objective_weights = np.ones(self.objectives) / self.objectives   
        else:
            assert len(self.objective_weights) == self.objectives, \
                "optimizer.objective_weights list should contain number of elements equal to optimizer.objectives"
        
        if self.objective_labels is None:
            self.objective_labels = [f'obj_{o}' for o in range(self.objectives)]
        else:
            assert len(self.objective_labels) == self.objectives, \
                "optimizer.objective_labels list should contain number of strings equal to optimizer.objectives"
            
        if self.constraint_labels is None:
            self.constraint_labels = [f'cnstr_{c}' for c in range(self.constraints)]
        else:
            assert len(self.constraint_labels) == self.constraints, \
                "optimizer.constraint_labels list should contain number of strings equal to optimizer.constraints"

        assert (isinstance(self.number_of_processes, int) and self.number_of_processes > 0)\
               or self.number_of_processes == 'maximum', \
               "optimizer.number_of_processes should be positive integer or \'maximum\'"
        if self.number_of_processes == 'maximum':
            self.number_of_processes = multiprocessing.cpu_count()
        if self.number_of_processes > 1: 
            if self.monitoring == 'basic':
                self.log(f'Preparing a multiprocess pool for {self.number_of_processes} processes.')
            self.pool = multiprocessing.Pool(self.number_of_processes)

        if self.maximum_evaluations:
            assert isinstance(self.maximum_evaluations, int) and self.maximum_evaluations > 0, \
                "optimizer.maximum_evaluations should be positive integer"

        if self.maximum_stalled_iterations:
            assert isinstance(self.maximum_stalled_iterations, int) and self.maximum_stalled_iterations > 0, \
                "optimizer.maximum_stalled_iterations should be positive integer"
        
        if self.maximum_stalled_evaluations:
            assert isinstance(self.maximum_stalled_evaluations, int) and self.maximum_stalled_evaluations > 0, \
                "optimizer.maximum_stalled_evaluations should be positive integer"

        if not self.iterations:
            self.iterations = 100 * self.dimensions
        
        if self.eval_fail_behavior:
            assert self.eval_fail_behavior == 'abort' or \
                    self.eval_fail_behavior == 'retry' or \
                    self.eval_fail_behavior == 'ignore', \
                "optimizer.eval_fail_behavior should be 'abort'/'retry'/'ignore'"
        else:
            self.eval_fail_behavior = 'abort'
        
        if self.eval_retry_attempts:
            assert isinstance(self.eval_retry_attempts, int) and self.eval_retry_attempts > 0, \
                "optimizer.eval_retry_attempts should be positive integer"
        else:
            self.eval_retry_attempts = 10
        if self.eval_retry_recede:
            assert isinstance(self.eval_retry_recede, float) and 0 < self.eval_retry_recede < 1, \
                "optimizer.eval_retry_attempts should be positive number in range (0,1)"
        else:
            self.eval_retry_recede = 0.01

        assert isinstance(self.monitoring, str) or self.monitoring is None, \
            'optimizer.monitoring should be a string or None'

        if isinstance(self.monitoring, str):
            # Convert to lowercase string
            self.monitoring = self.monitoring.lower()
        if self.monitoring is None:
            self.monitoring = 'none'

        if self.monitoring == 'none':
            self._progress_log = lambda: \
                self.__convergence_log_line() if self.convergence_log_file else None

        elif self.monitoring == 'basic':
            # self.progress_log = self._progress_log2
            self.rich_console = Console(highlight=False)

            if self.convergence_log_file:
                self._progress_log = lambda: \
                    (self.__progress_log_basic(), self.__convergence_log_line())
            else:
                self._progress_log = self.__progress_log_basic

        elif self.monitoring == 'dashboard':
            self._progress_log = lambda: \
                self.__convergence_log_line() if self.convergence_log_file else None

            self.__progress_bar = Progress(
                "[progress.description]{task.description}",
                # "{task.completed:} of {task.total}",
                BarColumn(),
                "[progress.percentage]{task.percentage:>3.0f}%",
                # "Elapsed: {task.time_remaining}",
                "Elapsed:",
                TimeElapsedColumn(),
                "Remaining:",
                TimeRemainingColumn(),
                # expand=True,
            )

            self.__progress_bar_tasks = [self.__progress_bar.add_task("Iterations:",
                                                                      total=self.iterations)]
            if self.maximum_stalled_iterations:
                self._stalled_it = 0
                self.__progress_bar_tasks.append(self.__progress_bar.add_task("Stalled iterations:",
                                                                              total=self.maximum_stalled_iterations))
            if self.maximum_evaluations:
                self.__progress_bar_tasks.append(self.__progress_bar.add_task("Evaluations:",
                                                                              total=self.maximum_evaluations))            
            if self.maximum_stalled_evaluations:
                self._stalled_eval = 0
                self.__progress_bar_tasks.append(self.__progress_bar.add_task("Stalled evaluations:",
                                                                              total=self.maximum_stalled_evaluations))
            if self.target_fitness:
                self.__progress_bar_tasks.append(self.__progress_bar.add_task("Target fitness:", total=1))

            def update_progress_bar():

                progress = [self.it]
                report_str = f'\nCurrent iteration: {self.it}'

                if self.maximum_stalled_iterations:
                    progress.append(self._stalled_it)
                    report_str += f'\nStalled iterations: {self._stalled_it}'
                
                if self.maximum_evaluations:
                    progress.append(self.eval)
                    report_str += f'\nTotal evaluations: {self.eval}'
                
                if self.maximum_stalled_evaluations:
                    progress.append(self._stalled_eval)
                    report_str += f'\nStalled evaluations: {self._stalled_eval}'
                    # This is related to the Issue #29 reported in GitLab
                    # self.log(f'{self._stalled_eval}, {self.maximum_stalled_evaluations}')
                    # report_str += f'\nStalled evaluations: {np.min([self._stalled_eval, self.maximum_stalled_evaluations])}'

                if self.target_fitness and self.best:
                    f0 = None
                    # for h in self.results.cHistory:
                    #     if np.all(h[2].C <= 0):
                    #         f0 = h[2].f
                    #         break
                    # print(f'{f0=}')
                    for i in range(self.it + 1):
                        if np.all(self.history['C'][i, :] <= 0):
                            f0 = self.history['f'][i, 0]
                            break
                    # print(f'{f0=}')

                    if f0:
                        if self.best.f > self.target_fitness:
                            p = (self.best.f - self.target_fitness) / (f0 - self.target_fitness)
                        else:
                            p = 0
                        p = 1 - np.log10(1 + 9 * p)
                    else:
                        p = 0
                    progress.append(p)

                for task, prog in zip(self.__progress_bar_tasks, progress):
                    self.__progress_bar.update(task, completed=prog)

                best = 'None'
                if self.best is not None:
                    best = str(self.best.X)
                    all_labels = self.objective_labels + self.constraint_labels
                    label_len = np.max([len(lbl) for lbl in all_labels]) + 3
                    for o, o_label in zip(self.best.O, self.objective_labels):
                        best += '\n[magenta]' + f'{o_label}: '.rjust(label_len) + f'{o:12.5e} [/magenta]'
                    for c, c_label in zip(self.best.C, self.constraint_labels):
                        if c <= 0:
                            best += '\n[green]' + f'{c_label}: '.rjust(label_len) + f'{c:12.5e}   => (satisfied)[/green]'
                        else:
                            best += '\n[red]' + f'{c_label}: '.rjust(label_len) + f'{c:12.5e}   => (violated)[/red]'
                    best += '\n' + 'Fitness: '.rjust(label_len) + f'{self.best.f:12.5e}' + '\n'
                report_str += f'\nBest:\n{best}'
                
                return Group(f'[b]INDAGO[/b] {indago.__version__}',
                             f'{self.__class__.__name__} optimization running\n',
                             self.__progress_bar,
                             report_str)
            
            self.__update_progress_bar = update_progress_bar
            self.live = Live(self.__update_progress_bar())

        else:
            assert False, f'Unknown value for optimizer.monitoring parameter ({self.monitoring})'
        
        # Initialize optimization counters
        self.eval = 0
        self.it = 0

        # Initialize history
        self.history = {'eval': np.full([0, 1], 0),
                        'X': np.full([0, self.dimensions], np.zeros(self.dimensions)),
                        'O': np.full([0, self.objectives], np.zeros(self.objectives)),
                        'C': np.full([0, self.constraints], np.zeros(self.constraints)),
                        'f': np.full([0, 1], 0),
                        }
        # self.results = OptimizationResults(self)
        # self.results.cHistory = []
        self.best = None

    def _init_convergence_log(self):
        # Initialize convergence log file
        if self.convergence_log_file:
            if os.path.exists(self.convergence_log_file):
                os.remove(self.convergence_log_file)

            summary = ['#']
            summary.append(f'#  Indago {type(self).__name__} Optimizer')
            summary.append('#')
            for var, value in vars(self).items():
                if not var.startswith('_'):
                    if var in ['it', 'eval', 'history']:
                        continue
                    if isinstance(value, (int, float, str, bool)):
                        summary.append(f'#  {var:25}: {str(value)}')
                    if isinstance(value, (list, dict)) and len(value) > 0:
                        summary.append(f'#  {var:25}: {str(value)}')
                    if isinstance(value, np.ndarray) and np.size(value) > 0:
                        if isinstance(value[0], (int, float)):
                            summary.append(f'#  {var:25}: {np.array_str(value, max_line_width=np.inf)}')
            # for k, v in self.params.items():
            #     summary.append(f'#  {k:25}: {v}')

            summary.append('#')
            summary.append(f'#  Optimization started at {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}')
            summary.append('#')

            # Columns header
            line = 'Iterations'.rjust(12)
            line += 'Evaluations'.rjust(15)
            line += ''.join([f'x{i+1}'.rjust(20) for i in range(self.dimensions)])
            for ol in self.objective_labels:
                line += f'{ol}'[:19].strip().rjust(20)
            for cl in self.constraint_labels:
                line += f'{cl}'[:19].strip().rjust(20)
            line += 'Fitness'.rjust(20)

            log_file = open(self.convergence_log_file, 'w')
            log_file.write('\n'.join(summary) + '\n')
            log_file.write('#' + line + '\n')
            log_file.write('#' + '-' * len(line) + '\n')
            log_file.close()

    def _check_params(self, mandatory_params, optional_params, defined_params):
        """Private method which checks if optimizer parameters are defined in Optimizer.params dictionary.
        Should be called in initializations of derived Optimizer classes. It asserts if any of mandatory parameters is
        missing and prints a warning if unknown/excessive parameter is provided.

        Parameters
        ----------
        mandatory_params : list of str
            A list of mandatory parameter names/keys for given method.
        optional_params: list of str
            A list of optional parameter names/keys for given method.
        defined_params: list of str
            A list of method's parameters names/keys defined in Optimizer.params dictionary.

        Returns
        -------
        None
            Nothing
        """
        for param in mandatory_params:
            assert param in defined_params, f'Error: Missing parameter {param}'

        for param in defined_params:
            if param not in mandatory_params and param not in optional_params:
                print(f'Warning: Excessive parameter {param}')

        self._init_convergence_log()

    def __str__(self):
        """ Lazy method for a (hopefully) useful printout of optimizer properties """
        
        table = Table(title=f'Indago {type(self).__name__} Optimizer')

        table.add_column('Property', justify='left', style='magenta')
        table.add_column('Value', justify='left', style='cyan')
        
        for var, value in vars(self).items():
            if var in ['it', 'eval', 'history']:
                continue
            if not var.startswith('_'):
                if isinstance(value, (int, float, str, bool)):
                    table.add_row(var, str(value))
                if isinstance(value, (list, dict)) and len(value) > 0:
                    table.add_row(var, str(value))
                if isinstance(value, np.ndarray) and np.size(value) > 0:
                    if isinstance(value[0], (int, float)):
                        table.add_row(var, np.array_str(value, max_line_width=np.inf))
        
        # the following are important, explicitly adding them to printout
        if self.best:
            table.add_row('best.X', np.array_str(self.best.X, max_line_width=np.inf), style='bold')
            table.add_row('best.f', str(self.best.f), style='bold')
        
        console = Console()
        with console.capture() as capture:
            console.print(table)
        return capture.get()

    def log(self, msg, indent=0):
        """Utility function for consistently displaying messages in the output.

        Parameters
        ----------
        msg: str
            Message to be displayed in the output.
        indent: int
            Number of characters used for indentation of the message. Default is 0.
        Returns
        -------
        None
            Nothing
        """
        if self.monitoring == 'basic':
            print(' ' * indent + msg)
        if self.convergence_log_file:
            log_file = open(self.convergence_log_file, 'a')
            log_file.write(f'#  {msg}\n')
            log_file.close()

    def __progress_log_basic(self):
        line = f'iter: {self.it:{int(np.ceil(np.log10(self.iterations))) + 1}d}/{self.iterations}'
        line += f', eval: {self.eval:10d}'
        if self.maximum_evaluations:
            line += f'/{self.maximum_evaluations}'
        for io, ol in enumerate(self.objective_labels):
            line += f', {ol}: [cyan]{self.best.O[io]}[/cyan]'
        for cl, cv in zip(self.constraint_labels, self.best.C):
            if cv > 0:
                line += f', {cl}: [green bold]{cv}[/green bold]'
            else:
                line += f', {cl}: [green bold]{cv}[/green bold]'
        line += f', fit: [magenta bold]{self.best.f}[/magenta bold]'
        self.rich_console.print(line)
        #self.log(line)

    def __convergence_log_line(self):
        if not self.convergence_log_file:
            return

        line = f'{self.it}'.rjust(13)
        line += f'{self.eval}'.rjust(15)
        line += ''.join([f'{x:18.12e}'.rjust(20) for x in self.best.X])
        for o in self.best.O:
            line += f'{o:18.12e}'.rjust(20)
        for c in self.best.C:
            line += f'{c:18.12e}'.rjust(20)
        line += f'{self.best.f:18.12e}'.rjust(20)
        line += '\n'

        log_file = open(self.convergence_log_file, 'a')
        log_file.write(line)
        log_file.close()

    def tic(self):
        """Utility function for measuring the computational time. 
        Optimizer.tic() is used for starting the stopwatch.
        It is used in pair with Optimizer.toc() method which stops the stopwatch.

        Returns
        -------
        Float
            Return the recorded current time in seconds since the Epoch.
        """
        self._time1 = time.time()
        return self._time1

    def toc(self, msg='', silent=False):
        """Utility function for measuring the computational time. 
        Optimizer.toc() stops the stopwatch and returns time elapsed since 
        Optimizer.tic() was triggered.

        Parameters
        ----------
        msg: str, default=''
            A message which is displayed with the measured time. Not used if silent=True.
        silent: bool, default=True
            An option for disabling the output. Default is False. If True, the value of elapsed time is returned but no
            message is shown.

        Returns
        -------
        elapsed_time: float
            Time elapsed since calling the last Optimizer.toc() in seconds.
        """

        s = time.time() - self._time1
        
        if not silent:
            if msg == '':
                msg = 'Elapsed time:'

            if s < 1:
                dur = f'{s*1e3:.3f} milliseconds.'
            else:
                dur = f'{s:.3f} seconds.'
            self.log(f'{msg} {dur}')
            
        return s

    def _gen_unique_str(self):
        """A private method for generating a unique string of 16 characters used when forwarding a unique string to the
        evaluation function is enabled (Optimizer.forward_unique_str = True). The generated string is guaranteed to be
        unique within an optimizer.

        Returns
        -------
        s: str
            A unique string.
        """
        while True:
            _s = ''.join(np.random.choice([c for c in string.ascii_lowercase]) for _ in range(16))
            if _s not in self._unique_str_list:
                self._unique_str_list.append(_s)
                return _s

    def optimize(self, seed=None):
        """A method which starts the optimization. The method wraps :code:`_run` method of the optimizer's subclass.

        Parameters
        ----------
        seed: int or None
            A random seed. Use the same value for reproducing identical stochastic procedures.

        Returns
        -------
        optimum: OptimizationResults
            The best solution found in the optimization.
        """
        self._init_optimizer()
        np.random.seed(seed)

        if self.monitoring == 'dashboard':
            with self.live:
                r = self._run()
        else:
            r = self._run()

        if self.convergence_log_file:
            summary = ['#', f'#  Optimization finished at {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}', '#']
            log_file = open(self.convergence_log_file, 'a')
            log_file.write('\n'.join(summary))
            log_file.close()

        return r

    def _stopping_criteria(self):
        """A private method used to evaluate all specified stopping conditions.

        Returns
        -------
        stop: bool
            A :code:`True` or :code:`False`, whether the optimization should stop.
        """

        # Stop if fitness threshold achieved
        if self.target_fitness:
            if self.best.f < self.target_fitness and np.all(self.best.C <= 0):
                return True

        # Stop if maximum number of evaluations surpassed
        if self.maximum_evaluations:
            if self.eval > self.maximum_evaluations:
                return True
        
        # Stop if maximum number of stalled iterations surpassed
        if self.maximum_stalled_iterations:
            f_hist = self.history['f'].flatten()
            # f_hist = np.array([r[2].f for r in self.results.cHistory])
            self._stalled_it = np.size(np.where(f_hist == f_hist[-1])) - 1
            if self._stalled_it > self.maximum_stalled_iterations:
                if self.monitoring == 'basic':
                    self.log(f'Stalled for the last {self._stalled_it} iterations, optimization stopped')
                return True
            
        # Stop if maximum number of stalled evaluations surpassed
        if self.maximum_stalled_evaluations:
            f_hist = self.history['f'].flatten()
            # f_hist = np.array([r[2].f for r in self.results.cHistory])
            stalled_it = np.size(np.where(f_hist == f_hist[-1])) - 1
            # print(f'{stalled_it=}, {self.it=}')
            if stalled_it > self.it:
                self._stalled_eval = self.eval
            else:
                self._stalled_eval = self.eval - self.history['eval'][-stalled_it - 1][0]
            # print(f'new: {self.stalled_eval=}')
            # self.stalled_eval = self.eval - self.results.cHistory[-stalled_it][1]
            # print(f'old: {self.stalled_eval=}')
            if self._stalled_eval > self.maximum_stalled_evaluations:
                if self.monitoring == 'basic':
                    self.log(f'Stalled for the last {self._stalled_eval} evaluations, optimization stopped')
                return True
        
        # No criteria satisfied
        return False

    def __getstate__(self):
        self_dict = self.__dict__.copy()
        # del wrapped in try-except to avoid KeyError (no 'pool' in self_dict)
        if 'pool' in self_dict:
            del self_dict['pool']
        else:
            self.log('Unexpected behaviour! Please report this issue to authors. There is no \'pool\' key in self.__dict__' )
        # try:
        #     del self_dict['pool']
        # except:
        #     pass
        return self_dict

    def __setstate__(self, state):
        self.__dict__.update(state)

    def _multiprocess_evaluate(self, candidates):
        """A private method used for calling parallel evaluation of multiple candidates. It relies on multiprocessing
        pool using map and starmap methods. The objectives, constraints and fitness of candidates in the list is updated
        after the evaluation.

        Parameters
        ----------
        candidates: list of CandidateState
            A list of candidates to be evaluated.

        Returns
        -------
        err_msg: string or None
            Error message (if any).
        """
        
        err_msg = None
        
        if self.forward_unique_str:
            for p in range(len(candidates)):
                candidates[p].unique_str = self._gen_unique_str()
            result = self.pool.starmap(self._evaluation_function_safe,
                                       [tuple([c.X, c.unique_str]) for c in candidates])
        else:
            result = self.pool.map(self._evaluation_function_safe, [c.X for c in candidates])
        
        # if failed evaluation
        if np.isnan(result).any():
            self.eval_fail_count += np.sum(np.isnan(result))
            
            if self.eval_fail_behavior == 'ignore':
                # report to log evaluation failed
                if self.monitoring == 'basic':
                    self.log('evaluation function failed, ignoring')
            
            elif self.eval_fail_behavior == 'abort':
                # report to log evaluation failed
                if self.monitoring == 'basic':
                    self.log('evaluation function failed, aborting optimization')
                return 'EVALUATION FAILED.' 
            
            elif self.eval_fail_behavior == 'retry':                      
                for i in range(self.eval_retry_attempts):
                    # report to log commencing retry
                    if self.monitoring == 'basic':
                        self.log(f'evaluation function failed, attempting retry nr. {i+1}')            
                    if self.best:
                        goodX = self.best.X
                    else:
                        goodX = (self.lb + self.ub)/2
                    for p in np.nonzero(np.isnan(result))[0]:
                        candidates[p].X += self.eval_retry_recede * \
                                            (goodX - candidates[p].X)
                        if self.forward_unique_str:
                            candidates[p].evaluate(self._gen_unique_str())
                        else:
                            candidates[p].evaluate()
                        self.eval += 1
                        # if successful, transfer fitness info back to result
                        if not np.isnan(candidates[p].f):
                            if self.objectives == 1 and self.constraints == 0:
                                result[p] = candidates[p].f
                            else:
                                result[p] = (*candidates[p].O, *candidates[p].C)
                        else:
                            self.eval_fail_count += 1
                    if not np.isnan(result).any():
                        break
                else:
                    if self.eval_retry_attempts > 0:
                        # report to log retrying exhausted
                        if self.monitoring == 'basic':
                            self.log('evaluation function retry attempts exhausted, aborting optimization')
                        return 'TOO MANY FAILED EVALUATIONS.'

        if self.objectives == 1 and self.constraints == 0:
            # Fast evaluation
            for p in range(len(candidates)):
                candidates[p].f = result[p]
                candidates[p].O[0] = result[p]
        else:
            # Full evaluation
            for p in range(len(candidates)):
                for io in range(self.objectives):
                    candidates[p].O[io] = result[p][io]
                candidates[p].f = np.dot(candidates[p].O, self.objective_weights)
                for ic in range(self.constraints):
                    candidates[p].C[ic] = result[p][self.objectives + ic]
        
        # return error message if any
        if err_msg:
            return err_msg

    def collective_evaluation(self, candidates):
        """Function used for evaluation of multiple candidates which automatically conducts parallel or serial
        evaluation and forwards a unique string to the evaluation function. Evaluation is performed in-place and the
        candidates provided via argument are updated.

        Parameters
        ----------
        candidates: list of CandidateState
            A list of candidates to be evaluated.

        Returns
        -------
        err_msg: string or None
            Error message (if any).
        """
        
        err_msg = None
        
        n = len(candidates)
        self.eval += n

        if self.number_of_processes > 1:
            err_msg = self._multiprocess_evaluate(candidates)
        
        else:
            for p in range(n):
                if self.forward_unique_str:
                    candidates[p].unique_str = self._gen_unique_str()
                    candidates[p].evaluate(candidates[p].unique_str)
                else:
                    candidates[p].evaluate()
                    
                    # if failed evaluation
                    if np.isnan(candidates[p].f):
                        self.eval_fail_count += 1
                        
                        if self.eval_fail_behavior == 'ignore':
                            # report to log evaluation failed
                            if self.monitoring == 'basic':
                                self.log('evaluation function failed, ignoring')
                        
                        elif self.eval_fail_behavior == 'abort':
                            # report to log evaluation failed
                            if self.monitoring == 'basic':
                                self.log('evaluation function failed, aborting optimization')
                            return 'EVALUATION FAILED.' 
            
                        elif self.eval_fail_behavior == 'retry':                       
                            for i in range(self.eval_retry_attempts):
                                # report to log commencing retry
                                if self.monitoring == 'basic':
                                    self.log(f'evaluation function failed, attempting retry nr. {i+1}')
                                if self.best:
                                    goodX = self.best.X
                                else:
                                    goodX = (self.lb + self.ub)/2
                                candidates[p].X += self.eval_retry_recede * \
                                                    (goodX - candidates[p].X)
                                if self.forward_unique_str:
                                    candidates[p].evaluate(self._gen_unique_str())
                                else:
                                    candidates[p].evaluate()
                                self.eval += 1
                                if not np.isnan(candidates[p].f):
                                    break
                                else:
                                    self.eval_fail_count += 1
                            else:
                                if self.eval_retry_attempts > 0:
                                    # report to log retrying exhausted
                                    if self.monitoring == 'basic':
                                        self.log('evaluation function retry attempts exhausted, aborting optimization')
                                    return 'TOO MANY FAILED EVALUATIONS.'

        # Determine new best candidate state
        if self.best is None:
            self.best = np.sort(candidates)[0].copy()
        else:
            # self.best = np.min(np.append(candidates, [self.best])).copy()
            candidates_best = np.sort(candidates)[0]
            if candidates_best <= self.best:
                self.best = candidates_best.copy()

        # Post iterational processing
        if self.post_iteration_processing:
            _candidates = np.sort(candidates)
            self.post_iteration_processing(self.it, _candidates, self.best)


        # Record to history
        self._update_history()
        
        if self.monitoring == 'dashboard':
            self.live.update(self.__update_progress_bar())
        
        # return error message if any
        if err_msg:
            return err_msg

    def _update_history(self):

        if self.history['eval'].size < self.it + 1:
            self.history['eval'] = np.vstack([self.history['eval'], self.eval])
            self.history['X'] = np.vstack([self.history['X'], self.best.X])
            self.history['O'] = np.vstack([self.history['O'], self.best.O])
            self.history['C'] = np.vstack([self.history['C'], self.best.C])
            self.history['f'] = np.vstack([self.history['f'], self.best.f])
        else:
            self.history['eval'][-1] = self.eval
            self.history['X'][-1] = self.best.X
            self.history['O'][-1] = self.best.O
            self.history['C'][-1] = self.best.C
            self.history['f'][-1] = self.best.f

    def plot_history(self, axes=None):
        """Plot convergence

        Parameters
        ----------
        axes :

        Returns
        -------

        """

        fig = None
        if axes is None:
            fig, axes = plt.subplots(figsize=(6, 8), nrows=4, constrained_layout=True, sharex=True)
        ax_x, ax_o, ax_c, ax_f = axes

        E = self.history['eval'][:, 0]

        # Plot optimization variables
        if ax_x:
            for i in range(self.dimensions):
                X = self.history['X'][:, i]
                ax_x.plot(E, X, lw=0.5,
                          label=None if self.dimensions > 10 else f'$x:{i + 1}$')
            ax_x.set_ylabel('Optimizaion variables')
            if self.dimensions > 10:
                ax_x.legend()

        # Plot objectives
        if ax_o:
            for i in range(self.objectives):
                O = self.history['O'][:, i]
                ax_o.plot(E, O, lw=0.5, label=self.objective_labels[i])
            ax_o.set_ylabel('Objectives')
            ax_o.legend()

        # Plot constraints
        i_feasible = 0
        for i in range(self.it + 1):
            if np.all(self.history['C'][i, :] <= 0):
                i_feasible = i
                break
        if ax_c:
            for i in range(self.constraints):
                C = self.history['C'][:, i]
                ax_c.plot(E, C, lw=0.5, label=self.constraint_labels[i])
                I = C > 0
                ax_c.plot(E[I], C[I], 'r.')
            c_lim = ax_c.get_ylim()[0], ax_c.get_ylim()[1]
            ax_c.fill_between(E[:i_feasible], ax_c.get_ylim()[0], ax_c.get_ylim()[1],
                              color='red', alpha=0.2)
            ax_c.axhline(0, c='r', lw=0.5)
            ax_c.set_ylim(c_lim)
            ax_c.set_ylabel('Constraints')
            ax_c.legend()

        # Plot fitness
        if ax_f:
            F = self.history['f'][:, 0]
            ax_f.plot(E, F, lw=1, c='k')
            f_lim = ax_f.get_ylim()[0], ax_f.get_ylim()[1]
            ax_f.fill_between(E[:i_feasible], f_lim[0], f_lim[1],
                              color='red', alpha=0.2)
            ax_f.set_ylim(f_lim)
            ax_f.set_ylabel('Fitness')
            ax_f.set_xlabel('Evaluations')

        if fig:
            fig.align_ylabels(axes)

class CandidateState:
    """Candidate solution for optimization problem"""
    
    def __init__(self, optimizer: Optimizer):
        self._optimizer = optimizer
        self.X = np.zeros(optimizer.dimensions)
        self.O = np.zeros(optimizer.objectives) * np.nan
        self.C = np.zeros(optimizer.constraints) * np.nan
        self.f = np.nan
                    
        # Evaluation
        if self._optimizer.objectives == 1 and self._optimizer.constraints == 0:
            self._evaluate = self._eval_fast
        else:
            self._evaluate = self._eval_full
        
        # Comparison operators
        if self._optimizer.objectives == 1 and self._optimizer.constraints == 0:
            self._eq_fn = CandidateState._eq_fast
            self._lt_fn = CandidateState._lt_fast 
            #self.__gt__ = self._gt_fast
        else:
            self._eq_fn = CandidateState._eq_full
            self._lt_fn = CandidateState._lt_full 
            #self.__gt__ = self._gt_full
                
    def copy(self):
        # IMPORTANT: the method does not seem to work properly
        cP = CandidateState(self._optimizer)        
        # Real copy for ndarrays
        cP.X = np.copy(self.X)
        cP.O = np.copy(self.O)
        cP.C = np.copy(self.C)
        cP.f = self.f
        return cP    
    
    def __str__(self):
        
        table = Table(title=f'Indago {type(self).__name__} (subclass of CandidateState)')

        table.add_column('Property', justify='left', style='magenta')
        table.add_column('Value', justify='left', style='cyan')
        
        for var, value in vars(self).items():
            if not var.startswith('_'):
                if isinstance(value, (int, float, str, bool)):
                    table.add_row(var, str(value))
                if isinstance(value, (list, dict)) and len(value) > 0:
                    table.add_row(var, str(value))
                if isinstance(value, np.ndarray) and np.size(value) > 0:
                    if isinstance(value[0], (int, float)):
                        table.add_row(var, np.array_str(value, max_line_width=np.inf))
        
        console = Console()
        with console.capture() as capture:
            console.print(table)
        return capture.get()
        
    
    # Equality operator
    def __eq__(self, other): 
        return self._eq_fn(self, other) 
    
    @staticmethod
    def _eq_fast(a, b): 
        return a.f == b.f

    @staticmethod
    def _eq_full(a, b):
        # return np.sum(np.abs(a.X - b.X)) + np.sum(np.abs(a.O - b.O)) + np.sum(np.abs(a.C - b.C)) == 0.0
        return (a.X == b.X).all() and (a.O == b.O).all() and (a.C == b.C).all() and a.f == b.f

    # Inequality operator
    def __ne__(self, other):
        return self.f != other.f
    
    # Less-then operator
    def __lt__(self, other):
        return self._lt_fn(self, other)    

    def evaluate(self, s=None):
        return self._evaluate(s)

    @staticmethod
    def _lt_fast(a, b): 
        #print('_lt_fast')
        if np.isnan(a.f).any():
            return False
        if np.isnan(b.f).any():
            return True  
        return a.f < b.f   
    
    @staticmethod     
    def _lt_full(a, b):  
        #print("_lt_full")
        if np.isnan([*a.O, *a.C]).any():
            return False
        if np.isnan([*b.O, *b.C]).any():
            return True            
        if np.sum(a.C > 0) == 0 and np.sum(b.C > 0) == 0: 
            # Both are feasible
            # Better candidate is the one with smaller fitness
            return a.f < b.f
        
        elif np.sum(a.C > 0) == np.sum(b.C > 0): 
            # Both are unfeasible and break same number of constraints
            # Better candidate is the one with smaller sum of unfeasible (positive) constraint values
            return np.sum(a.C[a.C > 0]) < np.sum(b.C[b.C > 0])
        
        else:
            # The number of unsatisfied constraints is not the same
            # Better candidate is the one which breaks fewer constraints
            return np.sum(a.C > 0) < np.sum(b.C > 0)
       
    
    def __gt__(self, other):
        #print('__gt__')
        #return self.f > other.f
        return not (self._lt_fn(self, other) or self._eq_fn(self, other))
    """
    def _gt_fast(self, other):
        return self.f > other.f
    def _gt_full(self, other): 
        return not (self.__eq__(other) or self.__lt__(other))
    """    
    
    def __le__(self, other):
        return self._lt_fn(self, other) or self.__eq__(other)
    
    def __ge__(self, other):
        return self.__gt__(other) or self.__eq__(other)

    def set_X(self, X: np.ndarray):

        assert np.size(self.X) == np.size(X), 'Unexpected size of optimization vector X'
        self.X = X
        
    def _eval_full(self, s=None):

        if s is None:
            oc = self._optimizer._evaluation_function_safe(self.X)
        else:
            oc = self._optimizer._evaluation_function_safe(self.X, s)

        for io in range(self._optimizer.objectives):
            self.O[io] = oc[io]
        
        self.f = np.dot(self.O, self._optimizer.objective_weights)
        
        for ic in range(self._optimizer.constraints):
            self.C[ic] = oc[self._optimizer.objectives + ic]

        #return None #self.O, self.C, self.f
        
    def _eval_fast(self, s=None):

        if s is None:
            f = self._optimizer._evaluation_function_safe(self.X)
        else:
            f = self._optimizer._evaluation_function_safe(self.X, s)

        self.f = f
        self.O[0] = f
        
        #return None #self.O, self.C, self.f

#
# class OptimizationResults:
#     """Data holder for optimization results."""
#
#     def __init__(self, optimizer: Optimizer):
#
#         self.optimizer = optimizer
#         self.cHistory = []
#
#     def plot_convergence(self, axes=None):
#         """Plot convergence
#
#         Parameters
#         ----------
#         axes :
#
#         Returns
#         -------
#
#         """
#
#         if axes is None:
#             fig = plt.figure(constrained_layout=True)
#             spec = gridspec.GridSpec(ncols=1, nrows=2, figure=fig)
#             axo = fig.add_subplot(spec[0])
#             axc = fig.add_subplot(spec[1], sharex=axo)
#         else:
#             axo, axc = axes
#
#         for io in range(self.optimizer.objectives):
#             O = np.array([[ev, cB.O[io]] for (it, ev, cB) in self.cHistory])
#             axo.plot(O[:, 0], O[:, 1], label=self.optimizer.objective_labels[io], lw=1)
#
#         F = np.array([[ev, cB.f] for (it, ev, cB) in self.cHistory])
#         axo.plot(F[:, 0], F[:, 1], label='f', lw=2, ls='-', c='b')
#         axo.legend()
#
#         for ic in range(self.optimizer.constraints):
#             C = np.array([[ev, cB.C[ic]] for (it, ev, cB) in self.cHistory])
#             axc.plot(C[:, 0], C[:, 1], label=self.optimizer.constraint_labels[ic], lw=1)
#
#             IX = C[:, 1] > 0
#             C = C[IX, :]
#
#             # nc = np.sum(C>0)
#             # axc.plot(I[:nc], C[:nc], c='r', ls='-', lw=2)
#             # if nc == 1:
#             #     axc.plot(I[:nc], C[:nc],'ro')
#
#         axc.legend()
#         axo.set_ylabel('Objectives')
#         axc.set_xlabel('Evaluations')
#         axc.set_ylabel('Constraints')
#         #ax.set_yscale('log')
        
